/*
<Copyright Header>
Copyright (c) 2012 Jordan "Earlz" Earls  <http://lastyearswishes.com>
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. The name of the author may not be used to endorse or promote products
   derived from this software without specific prior written permission.
   
THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

This file is part of the MbedConsole project
*/

#ifndef PLEARLZ_H
#define PLEARLZ_H

#define MAXSTACK 64
#define MAXTEMPSTACK 64
#define MAXLINELENGTH 128
#define DICTIONARYSTEP 4 //how much memory to reserve when doing a reallocation for resizing
#define CODEBLOCKSTEP 16

enum ErrorType
{
    None,
    StackOverflow,
    StackUnderflow
};

//arguments are in the bytecode, not the stack! But, things like Add has no arguments and uses the stack
//pointers are assumed 32-bit?
enum Opcode
{
    BranchTrue, //argument is 16 bit vaddress
    BranchFalse, //argument is 16bit vaddress
    Branch,
    PushTemp, //pops from stack and pushes onto the temporary stack
    PopTemp, //pushes onto stack from temporary stack
    Push, //argument is 32bit number
    Drop, //trashes top value on stack
    Call, //call. argument is WORD entry pointer in dictionary
    CallInt, //call internal argument is function pointer
    Add,
    Sub,
    Mul,
    Div,
    Mod,
    Cgt, //takes two integers and compares for greater than pushes 1 if so, 0 if not
    Clt, //less than
    Cgte, //greater than or equal
    Clte,
    Ceq,
    Cneq,
    LoadStr, //argument is variable length string ending with 0. Pushes address onto stack
    LoadConst, //pushes the value pointed to by the argument
    StoreConst, //pops a value and stores it in the memory pointed to by the argument
    Load, //same as above, except the address is popped first. 
    Store,
    Swap, 
    New, //pushes a pointer to free memory for an integer
    NewVar, //pushes a pointer to memory. Size is specified by argument
    Delete, //pops a pointer and frees it. 
    Pick, //takes an integer position from the stack. Goes that many slots back in the stack and peeks at the value, and then pushes it on the top of the stack
    Ret //exit
};

typedef void (*BuiltinFunction)(void);
typedef struct
{
    char name[12];
    enum Type{
        Constant,
        ConstantString,
        Builtin,
        Function
    } type;
    union valueunion{
        int intvalue;
        BuiltinFunction builtin;
        char *string;
    } value;
} WordKey;

/**
BranchTarget tracks branches/labels. 
When a beginning conditional/looping structure is found, it adds a branch target to the list with `target` set to point to the relevant branch target in codeblock
When an ending structure is found, it replaces the branch target(which starts at -1) with the actual branch address,
**/
typedef struct TargetNode
{
    int target; //position inside of codeblock. Can't use pointers because life is cruel and the block could be reallocated at anytime. 
    TargetNode* previous; //previous instead of next because we're going to "destroy" this list in reverse
} BranchTarget;  

extern ErrorType pl_error;

int pl_shell();

static inline int is_whitespace(char c){
    return (c==' ') || (c=='\t') || (c=='\n') || (c=='\r');
}

static inline int is_numeric(char c){
    return (c>='0') && (c<='9');
}

static inline int is_alpha(char c){
    return ((c>='a') && (c<='z')) || ((c>='A') && (c<='Z'));
}

static inline int is_identifier(char c){
    return is_alpha(c) || is_numeric(c) || (c=='_');
}

static inline int is_quote(char c){
    return (c=='\"') || (c=='\'');
}
int forth_execute(uint8_t* block, int length);
WordKey *pl_lookup(char* name);
int pl_pop();
void pl_push(int val);



//builtin functions
void bi_print();


#endif