#include "Configuration.h"
#include "Log.h"

#define CFG_BUFF_SIZE   128

namespace MachineVision{
    
    Configuration::Configuration(void) {
        // Defaults
        this->use_dhcp = true;
        this->ip = "10.0.0.100";
        this->netmask = "255.255.255.0";
        this->gateway = "10.0.0.254";
        this->tcp_port = 6666;
    }
    
    bool Configuration::writeConfiguration(LocalFileSystem * fs, char * configfile) {
        // Create config
        ConfigFile cfg;
        char buffer[CFG_BUFF_SIZE];
        
        sprintf(buffer, "%d", this->useDhcp());        
        if (!cfg.setValue("DHCP", buffer)) {
            Log::e("Failed to set value for DHCP.\n");
            return false;
        }    
     
        if (!cfg.setValue("IP_ADDRESS", (char *)this->getIpAddress().c_str())) {
            Log::e("Failed to set value for IP_ADDRESS.\n");
            return false;
        }   
     
        if (!cfg.setValue("NETMASK", (char *)this->getNetmask().c_str())) {
            Log::e("Failed to set value for NETMASK.\n");
            return false;
        }    
     
        if (!cfg.setValue("GATEWAY", (char *)this->getGateway().c_str())) {
            Log::e("Failed to set value for GATEWAY.\n");
            return false;
        }    
     
        sprintf(buffer, "%d", this->getTcpPort());     
        if (!cfg.setValue("TCP_PORT", buffer)) {
            Log::e("Failed to set value for TCP_PORT.\n");
            return false;
        }    

        // Write config to local filesystem
        if (!cfg.write(configfile)) {
            Log::e("Failed to write configuration file\r\n");
            return false;
        }
        
        return true;
    }
    
    /**
     * Read config from filesystem.
     *
     * @param configfile The path to the config file
     */
    bool Configuration::readFromFile(LocalFileSystem * fs, char * configfile) {       
        // Read config from local filesystem
        ConfigFile cfg;
        if (!cfg.read(configfile)) {
            Log::e("Failure to read configuration file\r\n");
            return false;
        }

        // Parse config params
        char buffer[CFG_BUFF_SIZE];
        if (readConfigParameter(&cfg, "DHCP", buffer, CFG_BUFF_SIZE)){
            if (buffer[0] == '0'){   // No DHCP
                this->use_dhcp = false;
            } else {
                this->use_dhcp = true;
            }
        } else {
            Log::e("DHCP config could not be read\r\n");
            return false;
        }

        if (!this->use_dhcp) {      // Static IP
            if (readConfigParameter(&cfg, "IP_ADDRESS", buffer, CFG_BUFF_SIZE)){
                this->ip = std::string(buffer);
            } else {
                Log::e("IP_ADDRESS config could not be read\r\n");
                return false;
            }
            
            if (readConfigParameter(&cfg, "NETMASK", buffer, CFG_BUFF_SIZE)){
                this->netmask = std::string(buffer);
            } else {
                Log::e("NETMASK config could not be read\r\n");
                return false;
            }
            
            if (readConfigParameter(&cfg, "GATEWAY", buffer, CFG_BUFF_SIZE)){
                this->gateway = std::string(buffer);
            } else {
                Log::e("GATEWAY config could not be read\r\n");
                return false;
            }
        }
            
        if (readConfigParameter(&cfg, "TCP_PORT", buffer, CFG_BUFF_SIZE)){
            this->tcp_port = atoi(buffer);
        } else {
            Log::e("TCP_PORT config could not be read\r\n");
            return false;
        }
        
        return true;
    }
    
    bool Configuration::readConfigParameter(ConfigFile * cfg, char * key, char * value, int buffer_size) {
        if (cfg->getValue(key, &value[0], buffer_size)) {
            Log::v("Read config: '%s'='%s'\r\n", key, value);
            return true;
        } else {
            Log::e("Could not find config key: '%s'\r\n", key);
            return false;
        }
    }
    
    bool Configuration::useDhcp(void) {
        return this->use_dhcp;
    }
    
    std::string Configuration::getIpAddress(void) {
        return this->ip;
    }
    
    std::string Configuration::getNetmask(void) {
        return this->netmask;
    }
    
    std::string Configuration::getGateway(void) {
        return this->gateway;
    }
    
    int Configuration::getTcpPort(void) {
        return this->tcp_port;
    }
}


