/*
The MIT License (MIT)

Copyright (c) 2016 British Broadcasting Corporation.
This software is provided by Lancaster University by arrangement with the BBC.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation
the rights to use, copy, modify, merge, publish, distribute, sublicense,
and/or sell copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.
*/

#include "MicroBit.h"

const static char DEVICE_NAME[] = "MY_MICROBIT_BLE_MIDI";
const uint8_t MIDI_SERVICE[16] = {
    0x03, 0xb8, 0x0e, 0x5a, 0xed, 0xe8, 0x4b, 0x33, 0xa7, 0x51, 0x6c, 0xe3, 0x4e, 0xc4, 0xc7, 0x00
};
const uint8_t MIDI_CHAR[16] = {
    0x77, 0x72, 0xe5, 0xdb, 0x38, 0x68, 0x41, 0x12, 0xa1, 0xa9, 0xf2, 0x66, 0x9d, 0x10, 0x6b, 0xf3
};

MicroBit uBit;
Serial  pc(USBTX, USBRX);

bool isConnect = false;

//MIDI Charactarastics/Service
uint8_t             midiPayload[5] = { 0x80, 0x80, 0x00, 0x00, 0x00 };

GattCharacteristic  midiChar (MIDI_CHAR,
                              midiPayload, 5, 5,
                              GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);
GattCharacteristic *midiChars[] = {&midiChar, };
GattService        midiService(MIDI_SERVICE, midiChars,
                               sizeof(midiChars) / sizeof(GattCharacteristic *));

static Gap::ConnectionParams_t connectionParams;

//NoteOn
void notifyNote(uint8_t note, uint8_t velocity);

//BLE Connection Callback
void onConnectionCallback(const Gap::ConnectionCallbackParams_t *params)
{
    isConnect = true;
    pc.printf("connected. Got handle %u\r\n", params->handle);
    uBit.display.printChar('C');

    connectionParams.slaveLatency = 1;
    if (uBit.ble->gap().updateConnectionParams(params->handle, &connectionParams) != BLE_ERROR_NONE) {
        pc.printf("failed to update connection paramter\r\n");
    }
}

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    isConnect = false;
    pc.printf("Disconnected handle %u, reason %u\r\n", params->handle, params->reason);
    pc.printf("Restarting the advertising process\r\n");
    uBit.display.printChar('A');

    pc.printf("Start Advertising\r\n");
    uBit.ble->gap().startAdvertising();
}

int main()
{    
    // Initialise the micro:bit runtime.
    pc.printf("Initialising MicroBit\r\n");
    uBit.ble = new BLEDevice();
    uBit.init();
    uBit.ble->init();
    pc.printf("Init done\r\n");

    //Add conection and disconnection callback
    uBit.ble->gap().onConnection(onConnectionCallback);
    uBit.ble->gap().onDisconnection(disconnectionCallback);

    /* setup advertising */
    uBit.ble->gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    //Reverse UUID for Adv
    uint8_t TMP_ADV_SERVICE[16];
    for(int i = 0; i < 16; i++)
    {
        TMP_ADV_SERVICE[i] = MIDI_SERVICE[15-i];
    }
    uBit.ble->gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS, (uint8_t*)TMP_ADV_SERVICE, sizeof(TMP_ADV_SERVICE));
    uBit.ble->gap().accumulateAdvertisingPayload(GapAdvertisingData::GENERIC_COMPUTER);
    uBit.ble->accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    uBit.ble->gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    uBit.ble->setAdvertisingInterval(160); /* 100ms; in multiples of 0.625ms. */
    uBit.ble->startAdvertising();
    pc.printf("Start Advertising\r\n");
    uBit.display.printChar('A');

    uBit.ble->gattServer().addService(midiService);

    // Insert your code here!
    while(1) {
        if(isConnect) {
            for (int note = 0x1E; note < 0x5A; note ++) {
                notifyNote(note, 0x45);
                wait_ms(100);
                notifyNote(note, 0x00);
                wait_ms(100);
            }
        }
    }

    // If main exits, there may still be other fibers running or registered event handlers etc.
    // Simply release this fiber, which will mean we enter the scheduler. Worse case, we then
    // sit in the idle task forever, in a power efficient sleep.
    //release_fiber();
}

//NotifyNote
void notifyNote(uint8_t note, uint8_t velocity)
{
    //Send Note
    midiPayload[2] = 0x90;
    midiPayload[3] = note;
    midiPayload[4] = velocity;

    //Notify MIDI Data
    uBit.ble->gattServer().write(midiChar.getValueAttribute().getHandle(), midiPayload, sizeof(midiPayload));
}