#pragma once

#include <mbed.h>
#include "es_wifi.h"


#define WIFI_WRITE_TIMEOUT 3000
#define WIFI_READ_TIMEOUT 25000

class TCPConnector
{
private:
    ES_WIFIObject_t m_es_wifi_ctx;

public:

    TCPConnector() {}
    ~TCPConnector(){}

    bool wifi_connect(char *wifi_ssid, char* wifi_wpa_password, uint32_t max_attempts)
    {
        
        /* HW setup */
        if (ES_WIFI_RegisterBusIO(&m_es_wifi_ctx,
                                    SPI_WIFI_Init,
                                    SPI_WIFI_DeInit,
                                    SPI_WIFI_Delay,
                                    SPI_WIFI_SendData,
                                    SPI_WIFI_ReceiveData) != ES_WIFI_STATUS_OK)
            return false;

        if (ES_WIFI_Init(&m_es_wifi_ctx) != ES_WIFI_STATUS_OK)
            return false;
        ES_WIFI_ResetToFactoryDefault(&m_es_wifi_ctx);

        /* JOIN AP */
        for (int i = 0; i < max_attempts; i++)
        {
            if (ES_WIFI_Connect(&m_es_wifi_ctx, wifi_ssid, wifi_wpa_password, ES_WIFI_SEC_WPA_WPA2) == ES_WIFI_STATUS_OK)
            {
                return (ES_WIFI_GetNetworkSettings(&m_es_wifi_ctx) == ES_WIFI_STATUS_OK);
            }
            wait_ms(1000);
        }

        return false;      
    }


    bool tcp_connect(uint8_t socket, char *domain_name, uint16_t remote_port, bool secure,uint32_t max_attempts)
    {
        ES_WIFI_Conn_t conn;
        ES_WIFI_Status_t res;

        conn.Number = socket;
        conn.RemotePort = remote_port;
        conn.LocalPort = 0;
        conn.Type = (secure)?ES_WIFI_TCP_SSL_CONNECTION:ES_WIFI_TCP_CONNECTION;
        strncpy((char *)conn.RemoteHost, domain_name, sizeof(conn.RemoteHost));
        for(int i=0;i<max_attempts;i++)
        {
            res = ES_WIFI_ConnectToRemoteHost(&m_es_wifi_ctx, &conn);
            if(res == ES_WIFI_STATUS_OK) return true;
        }
        return false;
    }


    bool tcp_close(uint32_t socket)
    {
        ES_WIFI_Conn_t conn;
        conn.Number = socket;
        
        return (ES_WIFI_StopClientConnection(&m_es_wifi_ctx, &conn)== ES_WIFI_STATUS_OK);
    }


    bool tcp_write(uint8_t socket, char *data, uint32_t data_len)
    {
        uint16_t ack;
        uint16_t chunk_size;
        ES_WIFI_Status_t res;

        for(int i=0;i<data_len;i+=ES_WIFI_PAYLOAD_SIZE)
        {
            chunk_size = ((data_len-i)<ES_WIFI_PAYLOAD_SIZE)?(data_len-i):ES_WIFI_PAYLOAD_SIZE;
            res = ES_WIFI_SendData(&m_es_wifi_ctx, socket, (uint8_t*)data+i, chunk_size, &ack, WIFI_WRITE_TIMEOUT);
            if( res != ES_WIFI_STATUS_OK || chunk_size != ack ) return false;
        }
        return true;
    }


    bool tcp_read(uint8_t socket, char *rx_buffer, uint16_t rx_buffer_len, uint16_t *received_bytes,bool continued)
    {
        bool rx = false;
        *received_bytes = 0;

        rx = (continued)?( ES_WIFI_ReceiveData2(&m_es_wifi_ctx, (uint8_t*)rx_buffer, rx_buffer_len, received_bytes) == ES_WIFI_STATUS_OK ):
                         ( ES_WIFI_ReceiveData(&m_es_wifi_ctx, socket, (uint8_t*)rx_buffer, rx_buffer_len, received_bytes, WIFI_READ_TIMEOUT) == ES_WIFI_STATUS_OK );
        
        return rx;
    }





};