/* mbed Stepper Motor Library
 * Copyright (c) 2012 Fabio Durigon
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
 
#ifndef STEPPERMOTOR_H
#define STEPPERMOTOR_H

#include "mbed.h"

//!Library for handle Stepper Motor with a Power Module with Clock and Dir input

// Rotation direction
#define CW 1
#define CC 0

/** Library to control a Stepper Motor Driver Board with Clock and Dir input*/
class StepperMotorDrv
{
    public:
        /** Create a StepperMotorDrv Object
        *
        * @param PinDir Digital Output Pin to set Direction of motor
        * @param PinClk Pwm Output Pin for Clock signal
        */
        StepperMotorDrv(PinName PinDir, PinName PinClk);
        
        /** Destroys an instance of StepperMotorDrv Object*/
        ~StepperMotorDrv();
        
        /** Set duty cycle of clock pulse in Percentual
        *
        * @param PercentualValue Value 0-100%
        */
        void SetDutyCyclePerc(float PercentualValue);
        
        /** Set duty cycle of clock pulse in decimal value
        *
        * @param DecimalValue 0.0 to 1.0 (0.5 = 50%)
        */
        void SetDutyCycle(float DecimalValue);
        
        /** Set Motor Speed: frequency of clock pulse to apply on PinClk
        *
        * @param SpeedHz Frequency value in Hz
        */
        void SetSpeed(float SpeedHz);
        
        /** Set MaxSpeed parameter used only by "Perc functions"
        * to calculate raw frequency to apply.
        *
        * @param MaxSpeedHz Frequency in Hz
        *
        *
        * Example:
        * @code
        * SetMaxSpeed(1000); // Max Speed is 1000 Hz
        * SetSpeedPerc(50); // Set Speed to 50% that means 500 Hz
        * @endcode
        *
        *
        */
        void SetMaxSpeed(float MaxSpeedHz);
        
        /** Set Speed in Perc of MaxSpeed
        * @param PercValue 0 to 100 (in %)
        *
        *
        * Example:
        * @code
        * SetMaxSpeed(1000); // Max Speed is 1000 Hz
        * SetSpeedPerc(100); // Set Speed to 100% that means 1000 Hz
        * @endcode
        *
        *
        */
        void SetSpeedPerc(float PercValue);
        
        /** Increment Speed by Hz value
        * @param StepValueHz: amount of Hz to add to velocity
        *
        * @return new Speed in Hz
        */
        float SpeedUp(float ValueHz);
        
        /** Decrement Speed by Hz value
        * @param StepValueHz: amount of Hz to substract to velocity
        *
        * @return new Speed in Hz
        */
        float SpeedDown(float ValueHz);
        
        /** Set Step value to apply in SpeedUp() SpeedDown() like functions
        * @param StepValueHz: value in Hz
        *
        *
        * Example:
        * @code
        * SetStep(10); // Set Step to 10 Hz
        * ...
        * SpeedUp(); // Increase Speed by 10 Hz
        * ...
        * @endcode
        *
        *
        */
        void SetSpeedJump(float ValueHz);
        
        /** Increase velocity by a Step set with SetStep() function
        *
        * @return new Speed in Hz
        *
        */
        float SpeedUp(void);
        
        /** Decrease velocity by a Step set with SetStep() function
        *
        * @return new Speed in Hz
        *
        */
        float SpeedDown(void);
               
        /** Get Current Speed in Hz*/
        float GetSpeed(void);
        
        /** Get Current Speed in Percentual*/
        float GetSpeedPerc(void);
        
        /** Get MaxSpeed parameter set (in Hz)*/
        float GetMaxSpeed(void);
        
        /** Get duty cycle value in Percentual*/
        float GetDutyCyclePerc(void);
        
        /** Get duty cycle in decimal value (Ex: 0.5 means 50%)*/
        float GetDutyCycle(void);
        
        /** Get step value for SpedUp/Down (in Hz)*/
        float GetSpeedJump(void);
        
        /** Set Rotation Direction
        * @param RotationDir 1 or CW for ClockWise rotation, 0 or CC for Counterclockwise rotation
        *
        * N.B. CW and CC depends from MotorBoard Specs
        *
        */
        void SetDir(bool RotationDir);
        
        /** Set Rotation Direction to CW*/
        void SetDirCW(void);
        
        /** Set Rotation Direction to CC*/
        void SetDirCC(void);
        
        /** Revert Direction*/
        void RevertDir(void);
        
        /** Get current Direction
        *
        * @return 1 for CW, o for CC
        */
        bool GetDir(void);
        
        /** Run Motor*/
        void Run(void);
        
        /** Stop Motor*/
        void Stop(void);
        
        /** Run <nrSteps> step in <RotationDir> direction*/
        void RunStep(int nrSteps, bool RotationDir);
        
        /** Run <nrSteps> step in current direction*/
        void RunStep(int nrSteps);
        
        /** Run one step in current direction*/
        void RunStep(void);
        
        /** Run one step in clockwise direction*/
        void RunStepCW(void);
        
        /** Run one step in counterclockwise direction*/
        void RunStepCC(void);
        
        /** Reset current position to 0
        * currPos=0
        * currDegree=0
        */
        void SetZero(void);
        
        /** Set internal <currPos> to Pos
        * be carefull if you use degrees funcions
        * this don't affect the degrees variables
        */
        void SetCurrPos(int Pos);
        
        /* Get current position in steps positive or negative relative to zero*/
        int GetCurrPos(void);
        
        /** Get current Period of duty cycle in seconds*/
        float GetPeriod(void);
        
        /** Set degrees of rotation by one step of motor
        * Set it if You want to use Degrees functions
        *
        *
        * Example:
        * @code
        * SetDegreePerStep(1.6); // 1.6 degrees for each motor step
        * SetZero();
        * RunStep(10); // move 1.6 * 10 = 16 degrees
        * printf("%f",GetCurrentDegree()); // prints "16"
        * ...
        * @endcode
        *
        *
        */
        void SetDegreePerStep(float Degrees);
        
        /** Get current "Degrees per one step" value*/
        float GetDegreePerStep(void);
        
        /** Get current position in Degrees*/
        float GetCurrDegree(void);
        
            
    private:
        DigitalOut Dir;     
        PwmOut Clock;
        float MaxSpeed;
        float Speed;
        float SpeedJump;
        float DutyCycle;
        int CurrPos;
        float tPeriod;
        float tUp;
        float tDown;
        float DegreesPerStep;
        float CurrDegree;
        
};
#endif