/* mbed R/C Servo Library
 *
 * Copyright (c) 2013 Fabio Durigon
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
  
#ifndef MBED_SERVO_H
#define MBED_SERVO_H

#include "mbed.h"

/** Library to control a Servo via pwm pin*/
class Servo{
    public:
        /** Create a Servo object
        *
        * @param Pin Pwm output pin
        */
        Servo(PinName Pin);
        
        /** Set servo position in range from 0.0 to 1.0
        *
        * @param range ablosute position from 0.0 to 1.0
        * 0.0 Full left position
        * 0.5 Centre position
        * 1.0 Full right position
        *
        * @return normalized position
        */
        float Write(float Pos);
        //! write() version for compatibility with Simon Ford's Servo Library
        float write(float Pos);
        
        /** Set servo position in Degrees where 0 is the centre position
        * @param Degrees Degrees inside the "Degrees Range"
        *
        * @return normalized Degrees
        *
        * @code
        * servo.WriteDegrees(-45); 
        * @endcode
        */
        float WriteDegrees(float Degrees);
        
        /** Set the pulsewidth in secs and measured degrees for this servo to reach the minimum position (max escursion to the left)
        *
        * @param PulseSec Pulsewitdh in sec
        * @param Degrees Degrees associated to pulsewidth
        */
        void DefinePulseMin(float PulseSec, float Degrees);
        
        /** Set the pulsewidth in secs and measured degrees for this servo to reach the maximum position (max escursion to the right)
        *
        * @param PulseSec Pulsewitdh in sec
        * @param Degrees Degrees associated to pulsewidth
        */
        void DefinePulseMax(float PulseSec, float Degrees);
        
        /** Set total range in secs for the entire excursion(from pulsemin to pulsemax) associated to degrees excursion.
        *   Central position of servo is PulseSec/2 or Degrres/2.
        *
        * @param TotPulseSec Pulsewitdh in sec from min position to max position
        * @param Degrees Total Degrees associated to TotPulseSec
        */
        void DefinePulseTotal(float TotPulseSec, float TotDegrees);
        
        /** Read current position
        *
        * @return Current position within range from 0.0 to 1.0
        */
        float Read(void);
        //! read() version for compatibility with Simon Ford's Servo Library
        float read(void);
        
        /** Read current position in BYTE number
        *
        * @return Current position within range from 0 to 255
        */
        unsigned char ReadByte(void);
        
        /** Read current Degrees
        *
        * @return Current position in Degrees within range of Degrees
        */
        float ReadDegrees(void);
        
        /** Read the current pulsemin
        *
        * @return Current pulsewith for min position
        */
        float ReadPulseMin(void);
        
        /** Read the current pulsemax
        *
        * @return Current pulsewith for max position
        */
        float ReadPulseMax(void);
        
        /** Read current degreemin
        *
        * @return Current Degrees associated to current pulsemin
        */
        float ReadDegreesMin(void);
        
        /** Read current degreemax
        *
        * @return Current Degrees associated to current pulsemax
        */
        float ReadDegreesMax(void);
        
        /**  Shorthand for the write and read functions */
        Servo& operator= (float range);
        Servo& operator= (Servo& rhs);
        operator float();
        
        void PrintValues(void);
        
        private:
            PwmOut pwmout;
            float currpos;
            float currdegrees;
            float pulsemin;
            float pulsemax;
            float pulserange;
            float degreesmin;
            float degreesmax;
            float degreesrange;
            float degreeshrange;
};
#endif