#ifndef MINISTM32_H
#define MINISTM32_H

#include "mbed.h"
#include "Protocols.h"
//#include "GraphicsDisplay.h"
#if DEVICE_PORTINOUT
/** 
* My MINI-STM32-V3.0 board is wired up to the display in a bizarre way
* Data lines are PC0-7 and PB8-15
* Using them with BUS16 is very slow, so this wrapper does more direct port access
*/
class MINISTM32 : public Protocols
{
 public:

    /** 
    * Create a MINISTM32 display interface
    * Pins are hard-wired
    * DB0 - PC0
    * DB1 - PC1
    * DB2 - PC2
    * DB3 - PC3
    * DB4 - PC4
    * DB5 - PC5
    * DB6 - PC6
    * DB7 - PC7
    * DB8 - N/C
    * DB9 - N/C
    * DB10 - PB8
    * DB11 - PB9
    * DB12 - PB10
    * DB13 - PB11
    * DB14 - PB12
    * DB15 - PB13
    * DB16 - PB14
    * DB17 - PB15
    * CS - PC8
    * RS - PC9
    * WR - PC10
    * RD - PC11
    * RESET - Connected to the board global reset line
    * EN - PC12
    */ 
    MINISTM32();

protected:
  
    /** Send 8bit command to display controller 
    *
    * @param cmd: byte to send  
    *
    */   
    virtual void wr_cmd8(unsigned char cmd);
    
    /** Send 8bit data to display controller 
    *
    * @param data: byte to send   
    *
    */   
    virtual void wr_data8(unsigned char data);
    
    /** Send 2x8bit command to display controller 
    *
    * @param cmd: halfword to send  
    * @note 2cycles using pins[7:0]
    */   
    virtual void wr_cmd16(unsigned short cmd);
    
    /** Send 2x8bit data to display controller 
    *
    * @param data: halfword to send   
    * @note 2cycles using pins[7:0], only gram write cmd uses pins[15:8]
    */   
    virtual void wr_data16(unsigned short data);
    
    /** Send 16bit pixeldata to display controller 
    *
    * @param data: halfword to send   
    * @note here using all pins[15:0]
    */   
    virtual void wr_gram(unsigned short data);
    
    /** Send same 16bit pixeldata to display controller multiple times
    *
    * @param data: halfword to send
    * @param count: how many
    * @note here using all pins[15:0]
    */   
    virtual void wr_gram(unsigned short data, unsigned int count);
    
    /** Send array of pixeldata shorts to display controller
    *
    * @param data: unsigned short pixeldata array
    * @param lenght: lenght (in shorts)
    * @note here using all pins[15:0]
    */   
    virtual void wr_grambuf(unsigned short* data, unsigned int lenght);
    
    /** Read 16bit pixeldata from display controller (with dummy cycle)
    *
    * @param convert true/false. Convert 18bit to 16bit, some controllers returns 18bit
    * @returns 16bit color
    */ 
    virtual unsigned short rd_gram(bool convert);
    
    /** Read 4x8bit register data (with dummy cycle)
    * @param reg the register to read
    * @returns data as uint
    * 
    */ 
    virtual unsigned int rd_reg_data32(unsigned char reg);
    
    /** Read 3x8bit ExtendedCommands register data
    * @param reg the register to read
    * @returns data as uint
    * @note EXTC regs (0xB0 to 0xFF) are read/write registers, for Parallel mode directly accessible in both directions
    */ 
    virtual unsigned int rd_extcreg_data32(unsigned char reg, unsigned char SPIreadenablecmd);
    
    /** ILI932x specific, does a dummy read cycle, number of bits is protocol dependent
    * for PAR protocols: a signle RD bit toggle
    * for SPI8: 8clocks
    * for SPI16: 16 clocks
    */   
    virtual void dummyread ();
    
    /** ILI932x specific, select register for a successive write or read
    *
    * @param reg register to be selected
    * @param forread false = a write next (default), true = a read next
    * @note forread only used by SPI protocols
    */   
    virtual void reg_select(unsigned char reg, bool forread =false);
    
    /** ILI932x specific, write register with data
    *
    * @param reg register to write
    * @param data 16bit data
    */   
    virtual void reg_write(unsigned char reg, unsigned short data);
    
    /** ILI932x specific, read register
    *
    * @param reg register to be read
    * @returns 16bit register value
    */ 
    virtual unsigned short reg_read(unsigned char reg);
    
    /** HW reset sequence (without display init commands)   
    */
    virtual void hw_reset();
    
    /** Set ChipSelect high or low
    * @param enable 0/1   
    */
    virtual void BusEnable(bool enable);
  
   

private:

    PortInOut _HI;
    PortInOut _LO;
    DigitalOut _CS; 
    DigitalOut _DC;
    DigitalOut _WR;
    DigitalOut _RD;
  
};
#endif
#endif