#include "mbed.h"
#include "rtos.h"

#include "Arial12x12.h"
#include "Arial24x23.h"
#include "ILI932x.h"
#include "TouchScreen.h"


#define LCD_X_RES 240
#define LCD_Y_RES 320

#define LCD_CS PH_0
#define LCD_RESET PH_1
#define LCD_RS PA_0
#define LCD_WR PA_1
#define LCD_RD PA_4


#define T_CLK  PC_10
#define T_DIN  PC_12
#define T_DOUT PC_11
#define T_CS   PC_8
#define T_IRQ  PB_9

/**
* My touch screen lies at a slight angle and hides some of the screen   
* Adjust this value as needed until the calibration dots are visible on your screen
**/
#define CAL_OFFSET 4

/**
* hand-crafted values for the touch-screen calibration
**/
#define TOUCH_X_MIN 540
#define TOUCH_X_MAX 3700
#define TOUCH_Y_MIN 340
#define TOUCH_Y_MAX 3656

unsigned short backgroundcolor=Black;
unsigned short foregroundcolor=White;

ILI932x* mySCREEN;
TouchScreen* touchScreen;
Serial *pc;

volatile bool done;
volatile int xt, yt;
volatile int xmin = 9999;
volatile int xmax = 0;
volatile int ymin = 9999;
volatile int ymax = 0;
    

void touchStart(TouchPosition p) {
    mySCREEN->fillcircle(p.screenX, p.screenY, 10, Blue) ;
}
    
void touchMove(TouchPosition p) {
    mySCREEN->fillcircle(p.screenX+10, p.screenY, 10, Green) ;
}
    
void touchEnd(TouchPosition p) {
    mySCREEN->fillcircle(p.screenX+20, p.screenY, 10, Red) ;
    done = true;
    xt = p.x;
    yt = p.y;
}

void cal(int x, int y) {
    done = false;
    mySCREEN->fillcircle(x, y, 4, Red) ;
    while(!done) {
        // wait
    }
    pc->printf("X=%d, Y=%d\r\n", xt, yt);
    if(xmin>xt) xmin=xt;
    if(xmax<xt) xmax=xt;
    if(ymin>yt) ymin=yt;
    if(ymax<yt) ymax=yt;
}

void print_calibration() {
    // We have max and min values corresponding to points inside the screen area
    // Work out the scaling factor on each axis, and adjust the origin to account for the 
    // offset used during calibration
    float xscale, yscale;
    int axmin, axmax, aymin, aymax;
    xscale = (xmax - xmin) / (LCD_X_RES - 2*CAL_OFFSET);
    axmin = xmin - CAL_OFFSET * xscale;
    axmax = xmax + CAL_OFFSET * xscale;

    yscale = (ymax - ymin) / (LCD_Y_RES - 2*CAL_OFFSET);
    aymin = ymin - CAL_OFFSET * yscale;
    aymax = ymax + CAL_OFFSET * yscale;

    pc->printf("#define TOUCH_X_MIN %u\r\n", axmin);
    pc->printf("#define TOUCH_X_MAX %u\r\n", axmax);
    pc->printf("#define TOUCH_Y_MIN %u\r\n", aymin);
    pc->printf("#define TOUCH_Y_MAX %u\r\n", aymax);
}

int main()
{
    pc = new Serial(USBTX, USBRX);
    pc->baud (9600);
    pc->printf("\n\nSystem Core Clock = %.3f MHZ\r\n",(float)SystemCoreClock/1000000);

    mySCREEN = new ILI932x(PAR_8, PortC, LCD_CS, LCD_RESET, LCD_RS, LCD_WR, LCD_RD,"myLCD");

    pc->printf("Screen id %d\r\n", mySCREEN->tftID);

    mySCREEN->set_orientation(0);
    mySCREEN->cls();                // clear the screen
    mySCREEN->locate(0,0);
    mySCREEN->background(backgroundcolor);    // set background to black
    mySCREEN->foreground(foregroundcolor);    // set chars to white
    mySCREEN->set_font((unsigned char*) Arial24x23);

    touchScreen = new TouchScreen(T_DIN, T_DOUT, T_CLK, T_CS, T_IRQ);
    touchScreen->setTouchStartHandler(&touchStart);
    touchScreen->setTouchMoveHandler(&touchMove);
    touchScreen->setTouchEndHandler(&touchEnd);
    touchScreen->setLCDGeometry(LCD_X_RES, LCD_Y_RES, TOUCHSCREEN_ORIENTATION_PORTRAIT | TOUCHSCREEN_ORIENTATION_ROTATED);
    touchScreen->setCalibration(TOUCH_X_MIN, TOUCH_X_MAX, TOUCH_Y_MIN, TOUCH_Y_MAX);
    
    while (1) {
        pc->printf("Calibrate..\r\n");
        cal(CAL_OFFSET, CAL_OFFSET);
        cal(LCD_X_RES-CAL_OFFSET, CAL_OFFSET);
        cal(CAL_OFFSET, LCD_Y_RES-CAL_OFFSET);
        cal(LCD_X_RES-CAL_OFFSET, LCD_Y_RES-CAL_OFFSET);
        print_calibration();
        Thread::wait(1000);
    }
}
