#include "TextWidget.h"

/**
* A basic widget implementation which just draws some text.
* If the text does not fit in the bounding-box it will be clipped
**/
TextWidget::TextWidget(GraphicsContext *context) :
    Widget(context),
    _renderer(context->fontRenderer()),
    _font(context->defaultFont()),
    _halign(LEFT), _valign(TOP)
{
}

TextWidget::TextWidget(GraphicsContext *context, FontRenderer* renderer) :
    Widget(context),
    _renderer(renderer),
    _font(context->defaultFont()),
    _halign(LEFT), _valign(TOP)
{
}

TextWidget::TextWidget(GraphicsContext *context, FontRenderer* renderer, Font* font) :
    Widget(context),
    _renderer(renderer),
    _font(font),
    _halign(LEFT), _valign(TOP)
{
}

void TextWidget::setText(char* text)
{
    _text = text;
    dirty();
}

void TextWidget::setFont(Font* font)
{
    _font = font;
    dirty();
}

Font *TextWidget::getFont() {
    return _font;
}

void TextWidget::setHAlign(HAlign alignment) {
    _halign = alignment;
    dirty();
}

void TextWidget::setVAlign(VAlign alignment) {
    _valign = alignment;
    dirty();
}

void TextWidget::_draw()
{
    Widget::_draw();
    
    /**
    * Figure out how many lines of text we have
    **/
    int numLines = 1;
    char *c = _text;
    while(*c != NULL) {
        if(*c == '\n') {
            numLines++;
        }
        c++;
    }
    
    /******************************************************************/
    /*   ---------------------------------            ^               */
    /*   |                               |            |               */
    /*   |                               |            |               */
    /*   |                               |       inner.height         */
    /*   |                               |            |               */
    /*   |                               |            |               */
    /*   |                               |            |               */
    /*   |                               |            |               */
    /*   ---------------------------------            |               */
    /******************************************************************/
    /**
    * We need a window as high as the font with it's origin:
    * VALIGN=TOP    : (0, 0)
    * VALIGN=MIDDLE : (0, inner.height/2 - numLines * font.height/2)
    * VALIGN=BOTTOM : (0, inner.height   - numLines * font.height)
    **/
    int offset=0;
    switch(_valign) {
        case TOP:    offset = 0; break;
        case MIDDLE: offset = (_inner.height - (numLines * _font->zoomedHeight()))/2; break;
        case BOTTOM: offset = (_inner.height - (numLines * _font->zoomedHeight())); break;
    }

    _renderer->setForeground(_fg);
    _renderer->setBackground(_bg);
    
    // Renderer window is only high enough for the number of lines to draw. 
    int h = _font->zoomedHeight() * numLines;
    // Clip to fit within the TextWidget inner
    if((h + offset) > _inner.height) {
        h = _inner.height - offset;
    }
    
    _renderer->window(_inner.x, _inner.y + offset, _inner.width, h, false);
    _renderer->puts(_text, display(), _font);
    display()->copy_to_lcd();
}