#ifndef FILEINI_INCLUDED
#define FILEINI_INCLUDED

#define MAX_PATH 256

/** 
  * @file
  */
typedef struct INIKEY INIKEY;
typedef struct INIKEY* LPINIKEY;


/**  A structure that stores a configuration key node. */
struct INIKEY {
    char* ptrKeyName;
    char* ptrValue;
    LPINIKEY lpikNext;
};


typedef struct INI INI;
typedef struct INI* LPINI;

/** A structure that represent a section node */
struct INI {
    char* ptrSectionName;
    LPINIKEY lpikKey;
    LPINI lpinNext;
};

/** Sets string characters to upper case.
  * @param ptrSource A pointer to the string to transform to upper case.
  */
int _strupr(char* ptrSource);

/** @memberof INI
  * Loads and parses an Ini file into memory.
  * @param ptrFile A pointer to a NULL-terminated string that contains the name of the file to load.
  * If this parameters is NULL, the function create a void Ini file in memory.
  * @return Returns a pointer to an INI structure that represents the loaded file. If a error occurs, the return value is NULL.
  */
LPINI IniLoad(char* ptrFileName);

/** @memberof INI
  * Gets a pointer to a section.
  * @param ptrSectionFile A pointer to a NULL-terminated string that stores the name of the needed section.
  * @param lpinIni A pointer to an INI structure created with IniLoad().
  * @return Returns a pointer to an INI structure that represents the needed section. If a error occurs, the return value is NULL.
  */
LPINI IniGetSection(char* ptrSectionName, LPINI lpinIni);

LPINIKEY IniGetKey(char* ptrVarName, LPINI lpinSection);
/** \fn LPINIKEY IniGetKey(char* ptrVarName, LPINI lpinSection);
    @memberof INI
    \brief Gets a pointer to a key.
    \param[in] ptrVarName A pointer to a NULL-terminated string that stores the name of the needed key.
    \param[in] lpinSection A pointer to an INI structure that represents a section (opened with IniGetSection() or IniCreateSection()).
    \return Returns a pointer to an INIKEY structure that represents the needed key. If a error occurs, the return value is NULL.
*/

int IniGetValue(char* ptrValue, int iLength, LPINIKEY lpikIniKey);
/** \fn int IniGetValue(char* ptrValue, int iLength, LPINIKEY lpikIniKey);
    @memberof INIKEY
    \brief Gets a key value.
    \param[out] ptrValue A pointer to a buffer that will receive the key content.
    \param[in] iLenght The length of the buffer pointed by ptrValue.
    \param[in] lpikIniKey A pointer to an INIKEY structure that represents a key (opened with IniGetKey() or IniCreateKey()).
    \return Returns 0 if no error occurs, 1 else.
*/

int IniSetValue(char* ptrValue, LPINIKEY lpikIniKey);
/** \fn int IniSetValue(char* ptrValue, LPINIKEY lpikIniKey);
    @memberof INIKEY
    \brief Sets a key value.
    \param[in] ptrValue A pointer to a NULL-terminated string that stores the value to be set.
    \param[in] lpikIniKey A pointer to an INIKEY structure that represents a key (opened with IniGetKey() or IniCreateKey()).
    \return Returns 0 if no error occurs, 1 else.
*/
LPINIKEY IniCreateKey(char* ptrKeyName, char* ptrValue, LPINI lpinSection);
/** @extends INIKEY
    \fn LPINIKEY IniCreateKey(char* ptrKeyName, char* ptrValue, LPINI lpinSection);
    \brief Creates a key.
    \param[in] ptrKeyName A pointer to a NULL-terminated string that stores the name of the key to be created.
    \param[in] ptrValue The length of the NULL-terminated string that stores the value of the key to be created.
    \param[in] lpinSection A pointer to an INI structure that represents a section (opened with IniGetSection() or IniCreateSection()).
    \return Returns a pointer to an INIKEY structure that represents the created key. If a error occurs, the return value is NULL.
*/
LPINI IniCreateSection(char* ptrSectionName, LPINI lpinIni);
/** @extends INI
    \fn LPINI IniCreateSection(char* ptrSectionName, LPINI lpinIni);
    \brief Creates a section.
    \param[in] ptrSectionName A pointer to a NULL-terminated string that stores the name of the section to be created.
    \param[in] lpinIni A pointer to an INI structure that represents a file (opened with IniLoad()).
    \return Returns a pointer to an INI structure that represents the created section. If a error occurs, the return value is NULL.
*/
int IniSave(char* ptrFileName, LPINI lpinIni);
/** @extends INI
    \fn int IniSave(char* ptrFileName, LPINI lpinIni);
    \brief Saves an Ini file .
    \param[in] ptrFileName A pointer to a NULL-terminated string that stores the name of the file to be saved.
    \param[in] lpinIni A pointer to an INI structure that represents a file (opened with IniLoad()).
    \returnsReturns 0 if no error occurs, 1 else.
*/
void IniDeleteSection(LPINI lpinSection);
/** @extends INI
    \fn void IniDeleteSection(LPINI lpinSection);
    \brief Deletes a section.
    \param[in] lpinSection A pointer to an INI structure that represents the section to be deleted (opened with IniGetSection() or IniCreateSection()).
    \returns No return value avaliable.
*/
void IniDeleteKey(LPINIKEY lpikKey);
/** @extends INIKEY
    \fn void IniDeleteKey(LPINIKEY lpikKey);
  *  \brief Deletes a key.
  *  \param[in] lpinSection A pointer to an INI structure that represents the key to be deleted (opened with IniGetKey() or IniCreateKey()).
  *  \return No return value avaliable.
  */
void IniFree(LPINI lpinIni);
/** @extends INI
     \fn void IniFree(LPINI lpinIni);
  *  \brief Frees an Ini file.
  * \param[in] lpinIni A pointer to an INI structure that represents the file (opened with IniLoad()) to be freed.
  *  \return No return value avaliable.
  */

#define INI_ZERO(a) a->ptrSectionName=NULL; \
        a->lpikKey=NULL; \
        a->lpinNext=NULL

#define INIKEY_ZERO(a) a->ptrKeyName=NULL; \
        a->ptrValue=NULL; \
        a->lpikNext=NULL

#define EXIT_FREE {fclose(ptrFile);return 0;}

#endif

