/**
 * @file app.h
 * @brief Application entry point
 * Copyright 2015 SEVENCORE Co., Ltd.
 *
 * @author HyeongJun Kim 
 * @version 1.0.0  
 * @date 2015-08-17
*/
#ifndef APP_H
#define APP_H

#include "BleMsgHandler.h"
#include "app_sw_version.h"
#include "diss_task.h"
#include "dialog_fota_config.h"

namespace sevencore_fota{

/**
 ****************************************************************************************
 * @addtogroup dialog_fota module
 * @brief Application entry point Header.
 *
 * @{
 ****************************************************************************************
 */

#define KE_IDX_GET(uint16_t) (((uint16_t) >> 8) & 0xFF)
#define KE_BUILD_ID(type, index) ( (uint16_t)(((index) << 8)|(type)) )

/*
 * DISS DEFINITIONS
 ****************************************************************************************
 */
/// Manufacturer Name (up to 18 chars)
#define APP_DIS_MANUFACTURER_NAME_STR       ("SevenCore")
#define APP_DIS_MANUFACTURER_NAME_STR_LEN   (9)
/// Model Number String (up to 18 chars)
#define APP_DIS_MODEL_NB_STR            ("SC-Lamp-02")
#define APP_DIS_MODEL_NB_STR_LEN        (10)
/// System ID - LSB -> MSB (FIXME)
#define APP_DIS_SERIAL_NB_STR               ("123-456")
#define APP_DIS_SERIAL_NB_STR_LEN           (7)
/// System ID - LSB -> MSB (FIXME)
#define APP_DIS_SYSTEM_ID               ("\x12\x34\x56\xFF\xFE\x9A\xBC\xDE")
#define APP_DIS_SYSTEM_ID_LEN           (8)

#define APP_DIS_SW_REV                  DA14583_REFDES_SW_VERSION
#define APP_DIS_FIRM_REV                DA14583_SW_VERSION
#define APP_DIS_FEATURES                (DIS_MANUFACTURER_NAME_CHAR_SUP | DIS_MODEL_NB_STR_CHAR_SUP | DIS_SYSTEM_ID_CHAR_SUP | DIS_SW_REV_STR_CHAR_SUP | DIS_FIRM_REV_STR_CHAR_SUP | DIS_PNP_ID_CHAR_SUP)
/// Advertising minimum interval
#define APP_ADV_INT_MIN   0x800
/// Advertising maximum interval
#define APP_ADV_INT_MAX   0x800
/// Advertising channel map
#define APP_ADV_CHMAP     0x07
/// Advertising data maximal length
#define APP_ADV_DATA_MAX_SIZE           (ADV_DATA_LEN - 3)
/// Scan Response data maximal length
#define APP_SCAN_RESP_DATA_MAX_SIZE     (SCAN_RSP_DATA_LEN)
#define APP_DFLT_ADV_DATA        "\x09\x03\xF0\x18\xFF\x18\x0A\x18\xF1\x18"
#define APP_DFLT_ADV_DATA_LEN    (8+2)
#define APP_SCNRSP_DATA         "\x09\xFF\x00\x60\x52\x57\x2D\x42\x4C\x45"
#define APP_SCNRSP_DATA_LENGTH  (10)
#define APP_DFLT_DEVICE_NAME    ("SC-Lamp-02")


typedef struct 
{
    unsigned char  free;
    struct bd_addr adv_addr;
    unsigned short conidx;
    unsigned short conhdl;
    unsigned char idx;
    unsigned char  rssi;
    unsigned char  data_len;
    unsigned char  data[ADV_DATA_LEN + 1];

} ble_dev;

#define MAX_SCAN_DEVICES 9 

enum
{
    /// Idle state
    APP_IDLE,
    /// Scanning state
    APP_CONNECTABLE,
    /// Connected state
    APP_CONNECTED,
    /// Number of defined states.
    APP_STATE_MAX,
    /// Scanning state
    APP_SCAN,
};

struct app_env_tag
{
    unsigned char state;
    unsigned char num_of_devices;
    ble_dev devices[MAX_SCAN_DEVICES];
};

extern struct app_env_tag app_env;

/**
 ****************************************************************************************
 * @brief Send Reset request to GAPM task.
 * @param[in] BMH     Ble Message Handler class reference
 * @return void.
 ****************************************************************************************
 */
void app_rst_gap(BleMsgHandler *BMH);
/**
 ****************************************************************************************
 * @brief Send enable request to DISS profile task.
 * @param[in] BMH     Ble Message Handler class reference
 * @return void.
 ****************************************************************************************
 */
void app_diss_db_create(BleMsgHandler* BMH);
/**
 ****************************************************************************************
 * @brief Send enable request to FOTA profile task.
 * @param[in] BMH     Ble Message Handler class reference
 * @return void.
 ****************************************************************************************
 */
void app_fota_server_db_create(BleMsgHandler* BMH);
/**
 ****************************************************************************************
 * @brief Send enable request to Fan Control profile task.
 * @param[in] BMH     Ble Message Handler class reference
 * @return void.
 ****************************************************************************************
 */
void app_fan_control_db_create(BleMsgHandler* BMH);
/**
 ****************************************************************************************
 * @brief Send enable request to Lamp Control profile task.
 * @param[in] BMH     Ble Message Handler class reference
 * @return void.
 ****************************************************************************************
 */
void app_lamp_control_db_create(BleMsgHandler* BMH);
/**
 ****************************************************************************************
 * @brief Send Start Advertising command to GAPM task.
 * @param[in] BMH     Ble Message Handler class reference
 * @return void.
 ****************************************************************************************
 */
void app_adv_start(BleMsgHandler* BMH);
/**
 ****************************************************************************************
 * @brief Set Bondabe mode.
 * @param[in] BMH     Ble Message Handler class reference
 * @return void.
 ****************************************************************************************
 */
void app_set_mode(BleMsgHandler* BMH);
/**
 ****************************************************************************************
 * @brief Send enable request to DISS profile task.
 * @param[in] device  Ble device info
 * @param[in] BMH     Ble Message Handler class reference
 * @return void.
 ****************************************************************************************
 */
void app_dis_enable(ble_dev *device,BleMsgHandler* BMH);
/**
 ****************************************************************************************
 * @brief Send enable request to FOTA profile task.
 * @param[in] device  Ble device info
 * @param[in] BMH     Ble Message Handler class reference
 * @return void.
 ****************************************************************************************
 */
void app_fota_server_enable(ble_dev *device,BleMsgHandler* BMH);
/**
 ****************************************************************************************
 * @brief Send enable request to fan_control profile task.
 * @param[in] device  Ble device info
 * @param[in] BMH     Ble Message Handler class reference
 * @return void.
 ****************************************************************************************
 */
void app_fan_control_enable(ble_dev *device,BleMsgHandler* BMH);
/**
 ****************************************************************************************
 * @brief Send enable request to Lamp_control profile task.
 * @param[in] device  Ble device info
 * @param[in] BMH     Ble Message Handler class reference
 * @return void.
 ****************************************************************************************
 */
void app_lamp_control_enable(ble_dev *device,BleMsgHandler* BMH);
/**
 ****************************************************************************************
 * @brief Send connection confirmation.
 * param[in] auth  Authentication requirements.
 * @param[in] device  Ble device info
 * @param[in] BMH     Ble Message Handler class reference
 * @return void.
 ****************************************************************************************
 */
void app_connect_confirm(uint8_t auth,ble_dev *device,BleMsgHandler* BMH);
/**
 ****************************************************************************************
 * @brief Send the GAPC_DISCONNECT_IND message to a task.
 * @param[in] dst     Task id of the destination task.
 * @param[in] conhdl  The conhdl parameter of the GAPC_DISCONNECT_IND message.
 * @param[in] reason  The reason parameter of the GAPC_DISCONNECT_IND message.
 * @param[in] BMH     Ble Message Handler class reference
 * @return void.
 ****************************************************************************************
 */
void app_send_disconnect(uint16_t dst, uint16_t conhdl, uint8_t reason,BleMsgHandler* BMH);

}//namespace

/// @} dialog_fota module

#endif//APP_H
