/**
 * @file BleMsgHandler.h
 * @brief Ble message handler ( Ble message is communication mathod between Mbed and DA14583 )
 * Copyright 2015 SEVENCORE Co., Ltd.
 *
 * @author HyeongJun Kim 
 * @version 1.0.0  
 * @date 2015-08-17
*/

#ifndef BLEMSGHANDLER_H
#define BLEMSGHANDLER_H

/**
 ****************************************************************************************
 * @addtogroup ext_fota module
 * @brief Ble message Handler Class Header.
 *
 * @{
 ****************************************************************************************
 */
 
#include "mbed.h"
#include "MsgQueue.h"
#include "SerialManager.h"
#include "gapc_task.h"
#include "gapm_task.h"

namespace sevencore_fota{

#define FIRMWARE_DATA_FRAGMENT_SIZE 1024

typedef struct {
    unsigned short bType;
    unsigned short bDstid;
    unsigned short bSrcid;
    unsigned short bLength;
} ble_hdr;    

class BleMsgHandler
{
public:
    /**
     ****************************************************************************************
     * @brief Ble message handler constructor only connected Device
     ****************************************************************************************
     */
    BleMsgHandler(Serial* _device);
    /**
     ****************************************************************************************
     * @brief Ble message handler constructor with connected Device & HostPC
     ****************************************************************************************
     */
    BleMsgHandler(Serial* _device, Serial* _hostpc);
    /**
     ****************************************************************************************
     * @brief Ble message handler destructor
     ****************************************************************************************
     */
    ~BleMsgHandler(void);
    /**
     ****************************************************************************************
     * @brief Ble message handler Start title print function
     ****************************************************************************************
     */
    void PrintTitle(void);
    /**
     ****************************************************************************************
     * @brief Create ble message
     ****************************************************************************************
     */
    void BleMsgAlloc(unsigned short id,unsigned short dest_id,unsigned short src_id,unsigned short data_len,void *pdata,uint8_t *msg);
    /**
     ****************************************************************************************
     * @brief Send ble message to device
     ****************************************************************************************
     */
    int BleSendMsg(uint8_t *msg, unsigned short size);
    /**
     ****************************************************************************************
     * @brief Receive ble message from device
     ****************************************************************************************
     */
    void BleReceiveMsg(void);
    /**
     ****************************************************************************************
     * @brief Get message from MsgQueue and Execute corresponding function
     ****************************************************************************************
     */
    void BleMsgHandle(void);
    /**
     ****************************************************************************************
     * @brief Debugging message output to hostpc
     ****************************************************************************************
     */
    void HostPcPrint(char *str);
    /**
     ****************************************************************************************
     * @brief Receive test method
     ****************************************************************************************
     */
    void ReceiveToSerialTest(void);
    /**
     ****************************************************************************************
     * @brief Receive and Store Da14583 flash data
     ****************************************************************************************
     */
    void FirmwareDataReceive(unsigned short code_size, char *version);
    
    
    
private:
    /**
     ****************************************************************************************
     * @brief GAPM Command Event Handler
     ****************************************************************************************
     */
    void HandleGapmCmpEvt(unsigned short msgid,struct gapm_cmp_evt *param,unsigned short dest_id,unsigned short src_id);
    /**
     ****************************************************************************************
     * @brief GAPC Command Event Handler
     ****************************************************************************************
     */
    void HandleGapcCmpEvt(unsigned short msgid,struct gapc_cmp_evt *param,unsigned short dest_id,unsigned short src_id);


private:    
    bool print_flag;
    Serial* device;
    Serial* hostpc;
    SerialManager* SerialM;
    MsgQueue* MsgQ;
    unsigned char recv_msg[512];
    FILE *fp;
};

}//namespace

/// @} ext_fota module
#endif //BLEMSGHANDLER_H