/**
 * @file SerialManager.cpp
 * @brief Serial Communication Management 
 * Copyright 2015 SEVENCORE Co., Ltd.
 *
 * @author HyeongJun Kim 
 * @version 1.0.0  
 * @date 2015-08-19
*/
#include "SerialManager.h"
/**
 ****************************************************************************************
 * @addtogroup ext_fota module
 * @brief Serial Manager Class Method Definition.
 *
 * @{
 ****************************************************************************************
 */
namespace sevencore_fota{

/**
 ****************************************************************************************
 * @brief Serial Manager constructor only connected Device.
 * @param[in] mbed serial class reference to device.
 ****************************************************************************************
 */
SerialManager::SerialManager(Serial *_device)
{
    print_flag = 0;
    device = _device;
    FE_MSG_PACKET_TYPE = 0x05;
    MAX_PACKET_LENGTH = 350;
    bReceiveState = 0;
    wDataLength = 0;
    wReceivePos = 0;
    bHdrBytesRead = 0;
    memset(bReceiveElementArr,0,512);
}
/**
 ****************************************************************************************
 * @brief Serial Manager constructor with connected Device & HostPC.
 * @param[in] mbed serial class reference to device.
 * @param[in] mbed serial class reference to hostpc.
 ****************************************************************************************
 */
SerialManager::SerialManager(Serial *_device, Serial *_hostpc)
{
    print_flag = 1;
    device = _device;
    hostpc = _hostpc;
    FE_MSG_PACKET_TYPE = 0x05;
    MAX_PACKET_LENGTH = 350;
    bReceiveState = 0;
    wDataLength = 0;
    wReceivePos = 0;
    bHdrBytesRead = 0;
    memset(bReceiveElementArr,0,512);
    PrintSerialManager();
}
/**
 ****************************************************************************************
 * @brief Serial Manager destructor.
 ****************************************************************************************
 */
SerialManager::~SerialManager(void)
{
}
/**
 ****************************************************************************************
 * @brief Serial Manager Start title print function.
 ****************************************************************************************
 */
void SerialManager::PrintSerialManager(void)
{
    if(print_flag == 1)
        hostpc->printf("\nSerialManager Start!!\n");
}
/**
 ****************************************************************************************
 * @brief Send ble message to serial port.
 * @param[in] Sending data pointer.
 * @param[in] Sending data size.
 ****************************************************************************************
 */
int SerialManager::SendToSerial(uint8_t *data,unsigned short size)
{
    int cnt = 0;
    if(print_flag == 1)
        hostpc->printf("\n>>> send size = %hu\n",size);
        
    for(int i=0; i < size; i++)
    {
        if( device->writeable())
        {
            if(print_flag == 1)
                hostpc->printf("%02X ", data[i]);
            device->putc(data[i]);
            cnt++;
        }
    }
    return cnt;    
}
/**
 ****************************************************************************************
 * @brief Receive ble message from serial port.
 * @param[in] Storing data pointer.
 * @return Boolean - msg_size (receive complete) or -1 (receive not complete)
 ****************************************************************************************
 */
int SerialManager::ReceiveToSerial(unsigned char *receive_msg) 
{
    int ret = -1;
    unsigned char tmp;
    if(device->readable())
    {
        tmp = device->getc();
        switch(bReceiveState)
        {
            case 0:
                if( tmp == FE_MSG_PACKET_TYPE )
                {
                    bReceiveState = 1;
                    wDataLength = 0;
                    wReceivePos = 0;
                    bHdrBytesRead = 0;
                    memset(bReceiveElementArr,0,512);
                    
                    bReceiveElementArr[wReceivePos] = tmp;
                    wReceivePos++;
                    if(print_flag == 1)
                        hostpc->printf("\n[Receiver] Packet Type |: %02X \n", tmp);
                }else
                {
                    if(print_flag == 1)
                        hostpc->printf("\n[Receiver] Packet Type Error |: %02X \n", tmp);
                }
                break;
            case 1:
                if(print_flag == 1)
                    hostpc->printf("R-%02X ",tmp);
                bHdrBytesRead++;
                bReceiveElementArr[wReceivePos] = tmp;
                wReceivePos++;
                if( bHdrBytesRead == 6 )
                    bReceiveState = 2;
                break;
            case 2:
                if(print_flag == 1)
                    hostpc->printf("R+%02X ",tmp);
                wDataLength += tmp;
                if( wDataLength > MAX_PACKET_LENGTH )
                    bReceiveState = 0;
                else
                {
                    bReceiveElementArr[wReceivePos] = tmp;
                    wReceivePos++;
                    bReceiveState = 3;
                }
                break;
            case 3:
                if( print_flag == 1 )
                    hostpc->printf("R=%02X ",tmp);
                wDataLength += (unsigned short) (tmp*256);
                if( wDataLength > MAX_PACKET_LENGTH )
                {
                    if( print_flag == 1 )
                        hostpc->printf("\n[Receiver] Over SIZE: %d \n", wDataLength);
                    bReceiveState = 0;
                }else if (wDataLength == 0)
                {
                    if( print_flag == 1 )
                        hostpc->printf("\n[Receiver] Zero SIZE: %d \n", wDataLength);
                    memcpy(receive_msg,bReceiveElementArr,wReceivePos);
                    ret = wReceivePos;
                    bReceiveState = 0;
                }else
                {
                    bReceiveElementArr[wReceivePos] = tmp;
                    wReceivePos++;
                    bReceiveState = 4;
                }
                break;
            case 4:
                if( print_flag == 1 )
                    hostpc->printf("R~%02X ",tmp);
                bReceiveElementArr[wReceivePos] = tmp;
                wReceivePos++;
                // 9 = 1(first byte = FE_MSG_PACKET_TYPE) + 2(type) +2(dstid) +2(srcid) +2(length size)
                if(wReceivePos == wDataLength + 9)
                {
                    memcpy(receive_msg,bReceiveElementArr,wReceivePos);
                    ret = wReceivePos;
                    bReceiveState = 0;
                    if( print_flag == 1 )
                        hostpc->printf("\n[Receiver] Rcv Data SIZE: %d \n", wDataLength);
                }
                break;
            default:
                if(print_flag == 1)
                    hostpc->printf("\nERROR STRAGE STATE R+%02X \n",tmp);
                break;
        }
    }
    
    return ret;   
}
/**
 ****************************************************************************************
 * @brief Receive data for size.
 * @param[in] Storing data pointer.
 * @param[in] transmision data size.
 ****************************************************************************************
 */
void SerialManager::DataReceive(uint8_t *databuf, unsigned short size)
{
    int i;
    
    for(i=0;i<size;i++)
    {
        databuf[i] = (uint8_t)device->getc();
    }
}
/**
 ****************************************************************************************
 * @brief Receive data test function.
 ****************************************************************************************
 */
void SerialManager::ReceiveToSerialTest(void) 
{
    unsigned char tmp;
    int pos = 0;
    char buff[1024];
    memset(buff,'K',1023);
    buff[1023] = '\0';
    hostpc->printf("\n pos = %d FIRST <%s> \n",pos,buff);
    tmp = device->getc();
    if( tmp == 0x80 ){
        tmp = device->getc();
        if( tmp == 0x46 )
        {
            hostpc->printf("\n FIRMWARE DATA CONFIRM!! \n");
        }
    }
    tmp = device->getc();//reserved value
    tmp = device->getc();//reserved value
    tmp = device->getc();//reserved value
    while(1)
    {
        if(device->readable())
        {
            tmp = device->getc();
            //hostpc->printf("%d ==> 0x%X\n",pos,tmp);
            buff[pos] = tmp;
            pos++;    
            if( pos%512 == 0 ){
                buff[1023] = '\0';
                hostpc->printf("\n pos = %d CHECK <%s> \n",pos,buff);
            }
            if( pos == 1024 ){
                pos = 0;
                memset(buff,'Z',1023);
                buff[1023] = '\0';
            }
        }
        
    }
    
    
}

}//namespace
/// @} ext_fota module
