#include "main.h"
#include "radio.h"

#if defined(SX127x_H) || defined(SX126x_H)
    #define BW_KHZ              500
    #define SPREADING_FACTOR    9
    #define CF_HZ              917300000
    #ifdef SX127x_H
        AnalogIn a1(A1);
    #else
        /* arduino analog pins used by radio shield */
        AnalogIn a1(PC_3);  // PC_2 and PC_4 also available
    #endif
#elif defined(SX128x_H)
    #define BW_KHZ              400
    #define SPREADING_FACTOR    9
    #define CF_HZ              2487000000
    /* arduino analog pins used by radio shield */
    AnalogIn a1(PC_3);  // PC_2 and PC_4 also available
#endif

DigitalOut myled(LED1);

Timer t;
volatile bool tx_done;
DigitalOut pc6_out(PC_6);

#define RX_TIMEOUT_US       200000
/**********************************************************************/


void txDoneCB()
{
    tx_done = true;
}

static uint16_t crc16( uint8_t *buffer, uint16_t length )
{
    uint16_t i;
    // The CRC calculation follows CCITT
    const uint16_t polynom = 0x1021;
    // CRC initial value
    uint16_t crc = 0x0000;

    if( buffer == NULL )
    {
        return 0;
    }

    for( i = 0; i < length; ++i )
    {
        uint16_t j;
        crc ^= ( uint16_t ) buffer[i] << 8;
        for( j = 0; j < 8; ++j )
        {
            crc = ( crc & 0x8000 ) ? ( crc << 1 ) ^ polynom : ( crc << 1 );
        }
    }

    return crc;
}

void tx_ack()
{
    uint8_t n = 0;
    uint16_t crc;

    if (Radio::radio.rx_buf[0] == CMD_OUT_PIN) {
        uint16_t ain = a1.read_u16();
        Radio::radio.tx_buf[n++] = CMD_OUT_PIN_ACK;
        /* TODO read analog pin and digital input pin */
        Radio::radio.tx_buf[n++] = ain;
        Radio::radio.tx_buf[n++] = ain >> 8;
    } else
        Radio::radio.tx_buf[n++] = 0xff;

    crc = crc16(Radio::radio.tx_buf, n);
    Radio::radio.tx_buf[n++] = crc;
    Radio::radio.tx_buf[n++] = crc >> 8;

    tx_done = false;
    Radio::Send(n, 0, 0, 0);
    printf("tx_ack\r\n");
    
    while (!tx_done)
        Radio::service();

    Radio::Rx(0);
}

volatile struct _rx_ {
    uint8_t length;
    float rssi;
    float snr;
} rx;

void rxDoneCB(uint8_t size, float Rssi, float Snr)
{
    rx.length = size;
    rx.rssi = Rssi;
    rx.snr = Snr;
}

void radio_tx(uint8_t* payload, uint8_t payload_len)
{
    int i;
    uint8_t n = 0;
    uint16_t crc;

    while (payload_len > 0) {
        //printf("n%u, paylen%u\r\n", n, payload_len);
        payload_len--;
        Radio::radio.tx_buf[payload_len] = payload[payload_len];
        n++;
    }

    crc = crc16(Radio::radio.tx_buf, n);
    Radio::radio.tx_buf[n++] = crc;
    Radio::radio.tx_buf[n++] = crc >> 8;

    for (i = 0; i < 3; i++) {
        int rx_timeout_at;
        int rx_start_at;

        tx_done = false;
        Radio::Send(n, 0, 0, 0);
        
        while (!tx_done)
            Radio::service();

        rx_start_at = t.read_us();
        rx_timeout_at = rx_start_at + RX_TIMEOUT_US;

        Radio::Rx(0);
        rx.length = 0;

        while (t.read_us() < rx_timeout_at) {
            Radio::service();
            if (rx.length > 0) {
                uint16_t rx_crc;
                printf(" rssi:%.1fdBm snr:%.1fdB ", rx.rssi, rx.snr);
                rx_crc = Radio::radio.rx_buf[rx.length-2];
                rx_crc |= Radio::radio.rx_buf[rx.length-1] << 8;
                crc = crc16(Radio::radio.rx_buf, rx.length-2);
                if (crc == rx_crc) {
                    printf("crcOk %u\r\n", i);
                    if (Radio::radio.rx_buf[0] == CMD_OUT_PIN_ACK) {
                        uint16_t ain = Radio::radio.rx_buf[1];
                        ain |= Radio::radio.rx_buf[2] << 8;
                        printf("ain %u\r\n", ain);
                    }
                    rx.length = 0;
                    return;
                } else
                    printf("crcFail %04x != %04x\r\n", crc, rx_crc);
                rx.length = 0;
            } 
        }
        printf("rx-timeout %u,  %u\r\n", i, t.read_us() - rx_start_at);
    } // ..for()
}

const RadioEvents_t rev = {
    /* Dio0_top_half */     NULL,
    /* TxDone_topHalf */    NULL,
    /* TxDone_botHalf */    txDoneCB,
    /* TxTimeout  */        NULL,
    /* RxDone  */           rxDoneCB,
    /* RxTimeout  */        NULL,
    /* RxError  */          NULL,
    /* FhssChangeChannel  */NULL,
    /* CadDone  */          NULL
};

int main()
{
    printf("\r\nreset\r\n");

    trigger_init();

    t.start();

    Radio::Init(&rev);

    Radio::Standby();
    Radio::LoRaModemConfig(BW_KHZ, SPREADING_FACTOR, 1);
    Radio::LoRaPacketConfig(8, false, true, false);  // preambleLen, fixLen, crcOn, invIQ
    Radio::SetChannel(CF_HZ);
    
    Radio::Rx(0);

    for (;;) {
        trigger_mainloop();

        if (rx.length > 0) {
            uint16_t crc, rx_crc;
            int i;
            for (i = 0; i < rx.length; i++) {
                printf("%02x ", Radio::radio.rx_buf[i]);
            }
            printf(" rssi:%.1fdBm, snr:%.1fdB\r\n", rx.rssi, rx.snr);
            rx_crc = Radio::radio.rx_buf[rx.length-2];
            rx_crc |= Radio::radio.rx_buf[rx.length-1] << 8;
            crc = crc16(Radio::radio.rx_buf, rx.length-2);
            if (crc == rx_crc) {
                bool sendAck = false;
                bool parsed = parse_radio_rx(Radio::radio.rx_buf);
                printf("crcOk\r\n");
                if (!parsed && Radio::radio.rx_buf[0] == CMD_OUT_PIN) {
                    pc6_out.write(Radio::radio.rx_buf[1]);
                    printf("out pin state: %u", Radio::radio.rx_buf[1]);
                    sendAck = true;
                } else if (parsed)
                    sendAck = true;

                printf("\r\n");

                if (sendAck)
                    tx_ack();
            } else
                printf("crc %04x != %04x\r\n", crc, rx_crc);

            rx.length = 0;
        } // ..if something received

        uart_service();

        Radio::service();
    } // ..for (;;)
}

