#include "hal/pinmap.h"
#include "PeripheralPins.h"
#include "i2c_device.h"
#include "smbus.h"
#include "mbed_interface.h" // mbed_error_printf

i2c_slave_t i2c;

static DMA_HandleTypeDef hdma_tx;
static DMA_HandleTypeDef hdma_rx;
static SMBUS_HandleTypeDef SmbusHandle;
static IRQn_Type event_i2cIRQ;
static IRQn_Type error_i2cIRQ;

static inline void put_cbuf(uint8_t o)
{
    i2c.cbuf[i2c.cbuf_in] = o;
    if (++i2c.cbuf_in == CBUF_SIZE)
        i2c.cbuf_in = 0;

    if (i2c.cbuf_in == i2c.cbuf_out) {
        i2c.c_overrun = true;
    }
}

static inline void err_irqHandler(uint32_t tmpisrvalue)
{
    uint32_t _icr = 0;
    if (tmpisrvalue & SMBUS_FLAG_BERR) {
        put_cbuf(CMD_BUSERR);
        put_cbuf(1);
        put_cbuf(hdma_tx.Instance->CNDTR); // req[0]

        _icr |= SMBUS_FLAG_BERR;
    }

    if (tmpisrvalue & SMBUS_FLAG_ARLO) {
        put_cbuf(CMD_ARLO);
        put_cbuf(1);
        put_cbuf(hdma_tx.Instance->CNDTR); // req[0]

        _icr |= SMBUS_FLAG_ARLO;
    }

    if (tmpisrvalue & SMBUS_FLAG_TIMEOUT) {
        SmbusHandle.Instance->CR1 &= ~I2C_CR1_TXDMAEN;
        __HAL_DMA_DISABLE(&hdma_tx);
        __HAL_UNLOCK(&hdma_tx);
        hdma_tx.State = HAL_DMA_STATE_READY;

        SmbusHandle.Instance->CR1 &= ~I2C_CR1_RXDMAEN;
        __HAL_DMA_DISABLE(&hdma_rx);
        __HAL_UNLOCK(&hdma_rx);
        hdma_rx.State = HAL_DMA_STATE_READY;

        put_cbuf(CMD_TIMEOUT);
        put_cbuf(1);
        put_cbuf(hdma_tx.Instance->CNDTR); // req[0]

        _icr |= SMBUS_FLAG_TIMEOUT;
    }

    __HAL_SMBUS_CLEAR_FLAG(&SmbusHandle, _icr);
}

#ifdef TARGET_STM32L4
void SMBUSx_ER_IRQHandler()
{
    err_irqHandler(SMBUS_GET_ISR_REG(&SmbusHandle));
}

void SMBUSx_EV_IRQHandler(void)
#else
void SMBUSx_IRQHandler(void)
#endif
{
    uint32_t tmpisrvalue = SMBUS_GET_ISR_REG(&SmbusHandle);
    uint32_t icr = 0;

    static bool i2cTransferDirection; // true = read
    static bool stopWaiting = false;

    static uint8_t _rx_buf[33]; // [0] is cmd

    /* often multiple flags are set simultaneously, yet must still be taken in correct order */

#ifdef DEBUG_SMBUS
    put_cbuf(CMD_ISR);
    put_cbuf(1);
    put_cbuf(tmpisrvalue & 0xff);
#endif /* DEBUG_SMBUS */

    if (tmpisrvalue & SMBUS_FLAG_ADDR) {
        i2cTransferDirection = SMBUS_GET_DIR(&SmbusHandle);
        if (i2cTransferDirection) { // if host read
            // 1: Read transfer, slave enters transmitter mode..  cmd has already been provided

            SmbusHandle.Instance->CR1 &= ~I2C_CR1_RXDMAEN;
            __HAL_DMA_DISABLE(&hdma_rx);
            __HAL_UNLOCK(&hdma_rx);
            hdma_rx.State = HAL_DMA_STATE_READY;
#ifdef TARGET_STM32L4
            // clear dma interrupt flags
            hdma_rx.DmaBaseAddress->IFCR = (DMA_ISR_GIF1 << hdma_rx.ChannelIndex);
#endif

            fill_tx_buf(_rx_buf[0]);

#ifdef DEBUG_SMBUS
            put_cbuf(CMD_ADDR_HOST_READ); // AHR
            put_cbuf(2);
            put_cbuf(cmd_to_length[_rx_buf[0]]);
            put_cbuf(hdma_tx.Instance->CNDTR);
#endif /* DEBUG_SMBUS */
            SmbusHandle.Instance->CR1 |= I2C_CR1_TXDMAEN;
            HAL_DMA_Start(&hdma_tx, (uint32_t)i2c.tx_buf, (uint32_t)&SmbusHandle.Instance->TXDR, cmd_to_length[_rx_buf[0]]);

            icr |= SMBUS_FLAG_ADDR;
        } else if (!stopWaiting) {
            // 0: Write transfer, slave enters receiver mode.

            SmbusHandle.Instance->CR1 |= I2C_CR1_RXDMAEN;
            HAL_DMA_Start(&hdma_rx, (uint32_t)&SmbusHandle.Instance->RXDR, (uint32_t)_rx_buf, sizeof(_rx_buf));

#ifdef DEBUG_SMBUS
            put_cbuf(CMD_ADDR_HOST_WRITE); // AHW
            put_cbuf(1);
            put_cbuf(hdma_rx.Instance->CNDTR);
#endif /* DEBUG_SMBUS */

            icr |= SMBUS_FLAG_ADDR;
        }

        stopWaiting = true;
    }

#ifdef TARGET_STM32L0
    err_irqHandler(tmpisrvalue);
#endif

    if (tmpisrvalue & SMBUS_FLAG_AF) {
        /* last byte the host wants was sent */
        SmbusHandle.Instance->CR1 &= ~I2C_CR1_TXDMAEN;
        __HAL_DMA_DISABLE(&hdma_tx);
        __HAL_UNLOCK(&hdma_tx);
        hdma_tx.State = HAL_DMA_STATE_READY;

#ifdef DEBUG_SMBUS
        put_cbuf(CMD_AF);
        put_cbuf(1);
        put_cbuf(hdma_tx.Instance->CNDTR);
#endif /* DEBUG_SMBUS */

        icr |= SMBUS_FLAG_AF;
    }

    if (tmpisrvalue & SMBUS_FLAG_STOPF) {
#ifdef DEBUG_SMBUS
        put_cbuf(CMD_STOPF);
        put_cbuf(0);
#endif /* DEBUG_SMBUS */

        icr |= SMBUS_FLAG_STOPF;
        if (i2cTransferDirection == 0) {
            /* host done writing to buf */
            unsigned i, len;

            SmbusHandle.Instance->CR1 &= ~I2C_CR1_RXDMAEN;
            __HAL_DMA_DISABLE(&hdma_rx);
            __HAL_UNLOCK(&hdma_rx);
            hdma_rx.State = HAL_DMA_STATE_READY;

            len = sizeof(_rx_buf) - hdma_rx.Instance->CNDTR;
            if (len > 0) {
                /* send to mainloop */
                put_cbuf(_rx_buf[0]);
                put_cbuf(len-1);
                for (i = 1; i < len; i++)
                    put_cbuf(_rx_buf[i]);
            }
        }

        if (!(tmpisrvalue & SMBUS_FLAG_TXE)) {
            /* slave wanted to send more than host wanted to receive */
            SmbusHandle.Instance->ISR = SMBUS_FLAG_TXE;
        }

        stopWaiting = false;
    } // ..if (tmpisrvalue & SMBUS_FLAG_STOPF)

    __HAL_SMBUS_CLEAR_FLAG(&SmbusHandle, icr);
}

#define I2C_TIMING              0x10900CD6

int smbus_init( PinName sda, PinName scl, uint8_t slaveAddress)
{
    DMA_Channel_TypeDef *dma_ch_tx, *dma_ch_rx;
    uint32_t dma_req_tx, dma_req_rx;
    RCC_PeriphCLKInitTypeDef  RCC_PeriphCLKInitStruct;
    uint32_t tmpisr;
    I2CName i2c;

    I2CName i2c_sda = (I2CName)pinmap_peripheral(sda, PinMap_I2C_SDA);
    I2CName i2c_scl = (I2CName)pinmap_peripheral(scl, PinMap_I2C_SCL);

    // Configure I2C pins
    pinmap_pinout(sda, PinMap_I2C_SDA);
    pinmap_pinout(scl, PinMap_I2C_SCL);
    pin_mode(sda, OpenDrainNoPull);
    pin_mode(scl, OpenDrainNoPull);

    i2c = (I2CName)pinmap_merge(i2c_sda, i2c_scl);;

    if (i2c == I2C_1) {
        SmbusHandle.Instance = I2C1;
        RCC_PeriphCLKInitStruct.PeriphClockSelection = RCC_PERIPHCLK_I2C1;
        RCC_PeriphCLKInitStruct.I2c1ClockSelection = RCC_I2C1CLKSOURCE_SYSCLK;
        __HAL_RCC_I2C1_CLK_ENABLE();
        event_i2cIRQ = I2C1_EV_IRQn;
        error_i2cIRQ = I2C1_ER_IRQn;

        __HAL_RCC_DMA1_CLK_ENABLE();
#ifdef TARGET_STM32L4
        dma_ch_tx = DMA1_Channel6;
        dma_ch_rx = DMA1_Channel7;
        dma_req_tx = DMA_REQUEST_3;
        dma_req_rx = DMA_REQUEST_3;
#elif defined(TARGET_STM32L0)
        dma_ch_tx = DMA1_Channel6;
        dma_ch_rx = DMA1_Channel7;
        dma_req_tx = DMA_REQUEST_6;
        dma_req_rx = DMA_REQUEST_6;
#endif
    }
#if defined I2C2_BASE
    else if (i2c == I2C_2) {
        SmbusHandle.Instance = I2C2;
        RCC_PeriphCLKInitStruct.PeriphClockSelection = RCC_PERIPHCLK_I2C2;
#ifdef RCC_I2C2CLKSOURCE_SYSCLK
        RCC_PeriphCLKInitStruct.I2c1ClockSelection = RCC_I2C2CLKSOURCE_SYSCLK;
#endif /* RCC_I2C2CLKSOURCE_SYSCLK */
        __HAL_RCC_I2C2_CLK_ENABLE();
        event_i2cIRQ = I2C2_EV_IRQn;
        error_i2cIRQ = I2C2_ER_IRQn;

        __HAL_RCC_DMA1_CLK_ENABLE();
#ifdef TARGET_STM32L4
        dma_ch_tx = DMA1_Channel4;
        dma_ch_rx = DMA1_Channel5;
        dma_req_tx = DMA_REQUEST_3;
        dma_req_rx = DMA_REQUEST_3;
#elif defined(TARGET_STM32L0)
        dma_ch_tx = DMA1_Channel4;
        dma_ch_rx = DMA1_Channel5;
        dma_req_tx = DMA_REQUEST_7;
        dma_req_rx = DMA_REQUEST_7;
#endif
    }
#endif /* I2C2_BASE */
#if defined I2C3_BASE
    else if (i2c == I2C_3) {
        SmbusHandle.Instance = I2C3;
        RCC_PeriphCLKInitStruct.PeriphClockSelection = RCC_PERIPHCLK_I2C3;
        RCC_PeriphCLKInitStruct.I2c3ClockSelection = RCC_I2C3CLKSOURCE_SYSCLK;
        __HAL_RCC_I2C3_CLK_ENABLE();
        event_i2cIRQ = I2C3_EV_IRQn;
        error_i2cIRQ = I2C3_ER_IRQn;

        __HAL_RCC_DMA1_CLK_ENABLE();
#ifdef TARGET_STM32L4
        dma_ch_tx = DMA1_Channel2;
        dma_ch_rx = DMA1_Channel3;
        dma_req_tx = DMA_REQUEST_3;
        dma_req_rx = DMA_REQUEST_3;
#elif defined(TARGET_STM32L0)
        dma_ch_tx = DMA1_Channel4;
        dma_ch_rx = DMA1_Channel5;
        dma_req_tx = DMA_REQUEST_14;
        dma_req_rx = DMA_REQUEST_14;
#endif
    }
#endif /* I2C3_BASE */
#if defined I2C4_BASE
    else if (i2c == I2C_4) {
        SmbusHandle.Instance = I2C4;
        RCC_PeriphCLKInitStruct.PeriphClockSelection = RCC_PERIPHCLK_I2C4;
        RCC_PeriphCLKInitStruct.I2c4ClockSelection = RCC_I2C4CLKSOURCE_SYSCLK;
        __HAL_RCC_I2C4_CLK_ENABLE();
        event_i2cIRQ = I2C4_EV_IRQn;
        error_i2cIRQ = I2C4_ER_IRQn;

#ifdef TARGET_STM32L4
        __HAL_RCC_DMA2_CLK_ENABLE();
        dma_ch_tx = DMA1_Channel1;
        dma_ch_rx = DMA1_Channel2;
        dma_req_tx = DMA_REQUEST_0;
        dma_req_rx = DMA_REQUEST_0;
#elif defined(TARGET_STM32L0)
        #error absent
#endif
    }
#endif /* I2C4_BASE */
    else {
        mbed_error_printf("unknown i2c %x\r\n", i2c);
        return -1;
    }

    HAL_RCCEx_PeriphCLKConfig(&RCC_PeriphCLKInitStruct);

    // use get_i2c_timing() from i2c_device.h?
    SmbusHandle.Init.Timing               = I2C_TIMING;
    SmbusHandle.Init.AnalogFilter         = SMBUS_ANALOGFILTER_ENABLE;
    SmbusHandle.Init.OwnAddress1          = slaveAddress;
    SmbusHandle.Init.AddressingMode       = SMBUS_ADDRESSINGMODE_7BIT;
    SmbusHandle.Init.DualAddressMode      = SMBUS_DUALADDRESS_DISABLE;
    SmbusHandle.Init.OwnAddress2          = 0x00;
    SmbusHandle.Init.GeneralCallMode      = SMBUS_GENERALCALL_DISABLE;
    SmbusHandle.Init.NoStretchMode        = SMBUS_NOSTRETCH_DISABLE;
    SmbusHandle.Init.PacketErrorCheckMode = SMBUS_PEC_DISABLE;
    SmbusHandle.Init.PeripheralMode       = SMBUS_PERIPHERAL_MODE_SMBUS_SLAVE;
    SmbusHandle.Init.SMBusTimeout         = I2C_TIMEOUTR_TIMOUTEN | 0x186;

    if (HAL_SMBUS_Init(&SmbusHandle) != HAL_OK)
    {
        return -1;
    }

    tmpisr = SMBUS_IT_ADDRI | SMBUS_IT_STOPI | SMBUS_IT_NACKI | SMBUS_IT_ERRI;
    __HAL_SMBUS_ENABLE_IT(&SmbusHandle, tmpisr);

#ifdef TARGET_STM32L4
    HAL_NVIC_SetPriority(event_i2cIRQ, 0, 0);
    HAL_NVIC_EnableIRQ(event_i2cIRQ);
    HAL_NVIC_SetPriority(error_i2cIRQ , 0, 0);
    HAL_NVIC_EnableIRQ(error_i2cIRQ);

    NVIC_SetVector(event_i2cIRQ, (uint32_t)SMBUSx_EV_IRQHandler);
    NVIC_SetVector(error_i2cIRQ, (uint32_t)SMBUSx_ER_IRQHandler);

#else
    HAL_NVIC_SetPriority(event_i2cIRQ, 0, 0);
    HAL_NVIC_EnableIRQ(event_i2cIRQ);

    NVIC_SetVector(event_i2cIRQ, (uint32_t)SMBUSx_IRQHandler);
#endif

      /* Configure the DMA handler for Transmission process */
    hdma_tx.Instance                 = dma_ch_tx;
    hdma_tx.Init.Request             = dma_req_tx;
    hdma_tx.Init.Direction           = DMA_MEMORY_TO_PERIPH;
    hdma_tx.Init.PeriphInc           = DMA_PINC_DISABLE;
    hdma_tx.Init.MemInc              = DMA_MINC_ENABLE;
    hdma_tx.Init.PeriphDataAlignment = DMA_PDATAALIGN_BYTE;
    hdma_tx.Init.MemDataAlignment    = DMA_MDATAALIGN_BYTE;
    hdma_tx.Init.Mode                = DMA_NORMAL;
    hdma_tx.Init.Priority            = DMA_PRIORITY_LOW;

    HAL_DMA_Init(&hdma_tx);

    hdma_rx.Instance                 = dma_ch_rx;
    hdma_rx.Init.Request             = dma_req_rx;
    hdma_rx.Init.Direction           = DMA_PERIPH_TO_MEMORY;
    hdma_rx.Init.PeriphInc           = DMA_PINC_DISABLE;
    hdma_rx.Init.MemInc              = DMA_MINC_ENABLE;
    hdma_rx.Init.PeriphDataAlignment = DMA_PDATAALIGN_BYTE;
    hdma_rx.Init.MemDataAlignment    = DMA_MDATAALIGN_BYTE;
    hdma_rx.Init.Mode                = DMA_NORMAL;
    hdma_rx.Init.Priority            = DMA_PRIORITY_HIGH;

    HAL_DMA_Init(&hdma_rx);

    return 0;
}

static inline uint8_t get_cbuf()
{
    uint8_t ret = i2c.cbuf[i2c.cbuf_out];
    if (++i2c.cbuf_out == CBUF_SIZE)
        i2c.cbuf_out = 0;

    return ret;
}

/* service_i2c(): call from main loop */
void service_i2c()
{
    if (i2c.cbuf_in != i2c.cbuf_out) {
        uint8_t buf[32];
        uint8_t n, cmd, len;
        cmd = get_cbuf();
        len = get_cbuf();
        /* host is writing: parse here */
        for (n = 0; n < len; n++)
            buf[n] = get_cbuf();

        /*if (len != cmd_to_length[cmd])
            pc.printf("[%02x != %02x] ", len, cmd_to_length[cmd]);*/

        service_i2c_write(cmd, len, buf);
    }
}

