#include "mbed_wait_api.h"
#include "demo.h"
#include <math.h>   /* trunc */


#define SAMPLE_ODR       ODR_LOW /*!< Accelerometer samples Output Data Rate */

static uint8_t verbose = 1; /* Verbose output to UART terminal ON/OFF. */


/**
 * @brief  Handle DEMO State Machine
 */
typedef enum
{
  STATUS_MEMS_INT1_DETECTED,
  STATUS_FIFO_DOWNLOAD,
  STATUS_SLEEP
} DEMO_FIFO_STATUS;

static void *LIS2DH12_0_handle = NULL;
static volatile DEMO_FIFO_STATUS demoFifoStatus;

static volatile uint8_t LPS22HH_DIL24_PRESENT = 0;
static void *LPS22HH_P_0_handle = NULL;
static void *LPS22HH_T_0_handle = NULL;

uint8_t accel_get_num_samples(void)
{
    uint16_t samplesToRead = 0;

    /* Get num of unread FIFO samples before reading data */
    if (BSP_ACCELERO_FIFO_Get_Num_Of_Samples_Ext(LIS2DH12_0_handle, &samplesToRead) == COMPONENT_ERROR)
    {
        return 0;
    }

    return samplesToRead;
}

static DrvStatusTypeDef Init_All_Sensors(void)
{
    DrvStatusTypeDef ret;

    ret = BSP_ACCELERO_Init(LIS2DH12_0, &LIS2DH12_0_handle);

    if (BSP_TEMPERATURE_Init(LPS22HH_T_0, &LPS22HH_T_0_handle) == COMPONENT_OK)
    {
        LPS22HH_DIL24_PRESENT = 1;
    } else {
        c_log_printf("\e[31mlps22hh temp fail\e[0m\r\n");
        ret = COMPONENT_ERROR;
    }

    if (BSP_PRESSURE_Init(LPS22HH_P_0, &LPS22HH_P_0_handle) == COMPONENT_OK)
    {
        LPS22HH_DIL24_PRESENT = 1;
    } else {
        c_log_printf("\e[31mlps22hh pres fail\e[0m\r\n");
        ret = COMPONENT_ERROR;
    }

    return ret;
}

int accel_is_enabled(uint8_t* status)
{
    DrvStatusTypeDef ret = BSP_ACCELERO_IsEnabled(LIS2DH12_0_handle, status);
    if (ret == COMPONENT_OK)
        return 0;
    else
        return -1;
}

int accel_enable(uint8_t en)
{
    DrvStatusTypeDef ret;
    if (en)
        ret = BSP_ACCELERO_Sensor_Enable(LIS2DH12_0_handle);
    else
        ret = BSP_ACCELERO_Sensor_Disable(LIS2DH12_0_handle);

    if (ret == COMPONENT_OK)
        return 0;
    else
        return -1;
}

static DrvStatusTypeDef Enable_All_Sensors(void)
{
    DrvStatusTypeDef ret = COMPONENT_NOT_IMPLEMENTED;

    if (LPS22HH_DIL24_PRESENT == 1)
    {
        BSP_TEMPERATURE_Sensor_Enable(LPS22HH_T_0_handle);
        ret = COMPONENT_OK;
    }
    if (LPS22HH_DIL24_PRESENT == 1)
    {
        BSP_PRESSURE_Sensor_Enable(LPS22HH_P_0_handle);
        ret = COMPONENT_OK;
    }
    
    return ret;
}

int demo_start()
{
    if (Init_All_Sensors() == COMPONENT_ERROR)
    {
        c_log_printf("Init_All_Sensors fail\r\n");
        return -1;
    }

    if (Enable_All_Sensors() == COMPONENT_ERROR)
    {
        c_log_printf("Enable_All_Sensors fail\r\n");
        return -1;
    }

    /* Configure LIS2DH12 Sensor for the DEMO application */
    if (BSP_ACCELERO_Set_ODR(LIS2DH12_0_handle, SAMPLE_ODR) == COMPONENT_ERROR)
    {
        c_log_printf("Set_ODR fail\r\n");
        return -1;
    }

    /* Set FIFO_OVR on INT1 */
    if (BSP_ACCELERO_FIFO_Set_INT1_FIFO_Overrun_Ext(LIS2DH12_0_handle, LIS2DH12_ENABLE) == COMPONENT_ERROR)
    {
        c_log_printf("Set_INT1 fail\r\n");
        return -1;
    }

    demoFifoStatus = STATUS_SLEEP;

    return 0;
}

int lis2dh_mainloop()
{
    uint8_t fifo_overrun_status;

    /* Handle DEMO State Machine */
    switch (demoFifoStatus)
    {
      case STATUS_MEMS_INT1_DETECTED:

        /* Check if FIFO is full */
        if (BSP_ACCELERO_FIFO_Get_Overrun_Status_Ext(LIS2DH12_0_handle, &fifo_overrun_status) == COMPONENT_ERROR)
        {
            c_log_printf("get_overrun_status fail\r\n");

            return LIS2DH_BSP_FAIL;
        }

        if (fifo_overrun_status == 1)
        {
          demoFifoStatus = STATUS_FIFO_DOWNLOAD;
        }
        else
        {
          demoFifoStatus = STATUS_SLEEP;
        }
        break;

      case STATUS_FIFO_DOWNLOAD:
        demoFifoStatus = STATUS_SLEEP;
        return LIS2DH_MAIN_READ_FIFO;

      case STATUS_SLEEP:
        /* Enter sleep mode */
        return LIS2DH_MAIN_SLEEP;

      default:
        return LIS2DH_FAIL_STATE;
    } // ..switch (demoFifoStatus)

    return LIS2DH_MAIN_NONE;
}

int lis2dh_set_fifo_mode()
{
    /* Set FIFO mode to FIFO */
    if (BSP_ACCELERO_FIFO_Set_Mode_Ext(LIS2DH12_0_handle, LIS2DH12_FIFO_MODE) == COMPONENT_ERROR)
    {
      return -1;
    }
    return 0;
}

int lis2dh_set_fifo_bypass()
{
    if (BSP_ACCELERO_FIFO_Set_Mode_Ext(LIS2DH12_0_handle, LIS2DH12_FIFO_BYPASS_MODE) == COMPONENT_ERROR)
        return -1;

    return 0;
}

void EXTI4_15_IRQHandler()
{
    HAL_GPIO_EXTI_IRQHandler(M_INT1_O_PIN);
    //HAL_GPIO_EXTI_IRQHandler(M_INT2_O_PIN);
}

void HAL_GPIO_EXTI_Callback(uint16_t GPIO_Pin)
{
    if (GPIO_Pin == M_INT1_O_PIN) {
        lis2dh_int1();
        demoFifoStatus = STATUS_MEMS_INT1_DETECTED;
    }
}

DrvStatusTypeDef lis2dh12_get_axes(SensorAxes_t *acceleration)
{
  /* Read single FIFO data (acceleration in 3 axes) */
  return BSP_ACCELERO_Get_Axes(LIS2DH12_0_handle, acceleration);
}

static void FloatToInt(float in, displayFloatToInt_t *out_value, int32_t dec_prec)
{
  if (in >= 0.0f)
  {
    out_value->sign = 0;
  }
  else
  {
    out_value->sign = 1;
    in = -in;
  }

  out_value->out_int = (int32_t)in;
  in = in - (float)(out_value->out_int);
  out_value->out_dec = (int32_t)trunc(in * pow(10, dec_prec));
}

static void Temperature_Sensor_Handler(void *handle, displayFloatToInt_t* outValue)
{
  uint8_t who_am_i;
  float odr;
  uint8_t id;
  float temperature;
  uint8_t status;

  BSP_TEMPERATURE_Get_Instance(handle, &id);

  BSP_TEMPERATURE_IsInitialized(handle, &status);

  if (status == 1)
  {
    if (BSP_TEMPERATURE_Get_Temp(handle, &temperature) == COMPONENT_ERROR)
    {
      temperature = 0.0f;
    }

    //displayFloatToInt_t out_value;
    FloatToInt(temperature, outValue, 2);
    c_log_printf("TEMP[%d]: %c%d.%02d\r\n", (int)id, ((outValue->sign) ? '-' : '+'),
             (int)outValue->out_int, (int)outValue->out_dec);

    if (verbose == 1)
    {
      if (BSP_TEMPERATURE_Get_WhoAmI(handle, &who_am_i) == COMPONENT_ERROR)
      {
        c_log_printf("\e[31mtemp-WHO AM I address[%d]: ERROR\e[0m\r\n", id);
      }
      else
      {
        c_log_printf("temp-WHO AM I address[%d]: 0x%02X\r\n", id, who_am_i);
      }

      if (BSP_TEMPERATURE_Get_ODR(handle, &odr) == COMPONENT_ERROR)
      {
        c_log_printf("\e[31mtemp-ODR[%d]: ERROR\e[0m\r\n", id);
      }
      else
      {
        displayFloatToInt_t out_value;
        FloatToInt(odr, &out_value, 3);
        c_log_printf("ODR[%d]: %c%d.%03d Hz\r\n", (int)id, ((out_value.sign) ? '-' : '+'),
                 (int)out_value.out_int, (int)out_value.out_dec);
      }
    }
  }
}

static void Pressure_Sensor_Handler(void *handle, displayFloatToInt_t* val)
{
  uint8_t who_am_i;
  float odr;
  uint8_t id;
  float pressure;
  uint8_t status;

  BSP_PRESSURE_Get_Instance(handle, &id);

  BSP_PRESSURE_IsInitialized(handle, &status);

  if (status == 1)
  {
    if (BSP_PRESSURE_Get_Press(handle, &pressure) == COMPONENT_ERROR)
    {

      c_log_printf("\e[31mGet_Press fail\e[0m\r\n", id);
      pressure = 0.0f;
    }

    //displayFloatToInt_t out_value;
    FloatToInt(pressure, val, 2);
    c_log_printf("PRESS[%d]: %c%d.%02d\r\n", (int)id, ((val->sign) ? '-' : '+'),
             (int)val->out_int, (int)val->out_dec);

    if (verbose == 1)
    {
      if (BSP_PRESSURE_Get_WhoAmI(handle, &who_am_i) == COMPONENT_ERROR)
      {
        c_log_printf("\e[31mpres-WHO AM I address[%d]: ERROR\e[0m\r\n", id);
      }
      else
      {
        c_log_printf("pres-WHO AM I address[%d]: 0x%02X\r\n", id, who_am_i);
      }

      if (BSP_PRESSURE_Get_ODR(handle, &odr) == COMPONENT_ERROR)
      {
        c_log_printf("\e[31mpres-ODR[%d]: ERROR\e[0m\r\n", id);
      }
      else
      {
        displayFloatToInt_t out_value;
        FloatToInt(odr, &out_value, 3);
        c_log_printf("ODR[%d]: %c%d.%03d Hz\r\n", (int)id, ((out_value.sign) ? '-' : '+'),
                 (int)out_value.out_int, (int)out_value.out_dec);
      }

    }
  }
}

void demo_sample_pressure(displayFloatToInt_t* x)
{
    Pressure_Sensor_Handler(LPS22HH_P_0_handle, x);
}

void demo_sample_temp(displayFloatToInt_t* outValue)
{
    Temperature_Sensor_Handler(LPS22HH_T_0_handle, outValue);
}


