/* Class SSEL_spi, Copyright 2014, David T. Mort (http://mbed.org/users/dtmort/)

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/
#ifndef MBED_SSEL_SPI_H
#define MBED_SSEL_SPI_H


#include "mbed.h"
extern "C" void TIMER2_IRQHandler (void);   //this method Linker loads ISR vector at compile time

/**A hardware implimentation of a Serial (chip) SElect line output designed primarily for SPI (Serial Peripheral Interface)
    *   devices by utilizing mbed LPC1768 Timer 2 hardware and it's inturrupt features.
    *
    *   Use of this class does require hardwiring p30 to the serial clock output (usually p7 or p13).
    *
    *   p8 is the SSEL serial select output pin (chip select) and is not relocatable.
    *
    *   Some SPI peripherals require data latching while the serial clock is still active on
    *   the last bit clocked-in, such as the Maxim MAX6952. Using the mbed SPI and DigitalOut API's 
    *   (for chip select) this timing is not possible. This SSEL_spi class takes advantage of 
    *   the speed achievable with the mbed SPI API, and leaves the chip select output handling
    *   entirely in the capable hands of the Timer 2 hardware with fully automated frame lengths 
    *   making code simpler to write and read.
    *
    *       - Flexible serial frame lengths from 0 .. 4,294,967,295 bits (usually a multiple of SPI bit length)
    *       - Un-attended automation of the chip select line.
    *       - Selectable mode for proper clock polarity and edge trigger (make same as SPI mode).
    *       - Selectable chip select polarity (active-low or active-high).
    *       - Latching data in while clock is still active on last bit,
    *         **such as required by the Maxim MAX6952. This is not possible with the mbed API, 
    *         onboard SPI or SSP hardware alone. SPI speed is limited up to 3Mhz for this feature. 
    *         Higher SPI speeds are permissable although due to the limitations of the hardware the chip 
    *         select will latch AFTER sclk returns to the inactive state, tolerated by some serial peripherals.
    *       - Allows one-shot loading of all serial daisy-chained hardware registers before latching. 
    *         Prevents display ghosting effect. For example 
    *         a daisy-chain of seven serial chips each requiring 16 bits each would set up a frame
    *         length of 7 x 16 = 112 bit frame length.
    * 
    *   The MAX6952 chip requires the following sequence:
    *
    *       [[http://datasheets.maxim-ic.com/en/ds/MAX6952.pdf]]
    *       -# Take CLK low.
    *       -# Take CS low. This enables the internal 16-bit shift register.
    *       -# Clock 16 bits of data into DIN, D15 first to D0 last, observing the setup and hold times. Bit D15 is low, indicating a write command.
    *       -# Take CS high **(while CLK is still high after clocking in the last data bit).
    *       -# Take CLK low.
    *
    * Example:
    * @code
    * // Send a 16 bit word to a SPI slave
    * // NOTE:  You must hardwire sclk output pin (usually p7 or p13)
    * //        to p30 which is the CAP2.0 sensing pin used by Timer2.
    *  
    * // Chip select is automatic and will always be p8
    *
    * #include "mbed.h"
    * #include "SSEL_spi.h"
    *
    * SPI      spi(p5,p6,p7); //mosi, miso, sclk
    * SSEL_spi cs(16,0);      //16 bits per frame, mode 0 (mode optional, 0 is default)
    *
    * int main(){
    * cs.active(1);           //optionally will establish active-high (1) chip select
    * spi.format(16,0);
    * spi.frequency(1000000);
    *
    * while(1) spi.write(0x5555); 
    *   
    * }
    * @endcode
    */
class SSEL_spi {    

public:
/**Create automatated SSEL chip select output for an SPI interface on p8 utilizing Timer2. Must hardwire SPI sclk output pin to p30.
    *
    *   @param frame Bit Length of the SSEL chip select frame. Usually a multiple of the SPI bit length transmission.
    *   @param mode Set up proper triggering from SPI sclk output. Default mode = 0. Set mode same as SPI mode.
    * 
    */
    SSEL_spi(uint32_t frame, int mode=0);

/** Set active state of the chip select pin 8
    *
    *  Is active-low (0) by default. Call only if change needed.
    *
    *   Can also be called to force an output high or low condition.
    *   @param cs 0 (default) for active-low, 1 for active-high
    */
    void active(bool cs);        
protected:

private:

};
#endif
