/*
* @author Jinhyuk Yoon
*
* Designed to control ESCON via NUCELO-F767ZI
*/

#include "ESCON_encoder.h"

ESCON_encoder::ESCON_encoder(PinName channelA_, PinName channelB_, int count_per_turn_) : channelA(channelA_), channelB(channelB_)  {
    
    pulses         = 0;
    revolutions    = 0;
    pos_in_degree  = 0.0;
    pos_in_radian  = 0.0;
    count_per_turn = count_per_turn_;

    //Workout what the current state is.
    int chanA = channelA.read();
    int chanB = channelB.read();

    //2-bit state.
    currState = (chanA << 1) | (chanB);
    prevState = currState;

    //X2 encoding, so uses interrupts on only channel A.
    channelA.rise(this, &ESCON_encoder::encode);
    channelA.fall(this, &ESCON_encoder::encode);
}

void ESCON_encoder::reset(void) {

    pulses        = 0;
    pos_in_degree = 0.0;
    pos_in_radian = 0.0;

}

int ESCON_encoder::getCurrentState(void) {

    return currState;

}

int ESCON_encoder::getPulses(void) {

    return pulses;

}

double ESCON_encoder::getPosInDegree(void) {

    pos_in_degree = pulses / (2 * double(count_per_turn)) * 360;
    return pos_in_degree;
    
}

double ESCON_encoder::getPosInRadian(void) {

    pos_in_radian = pulses / (2 * double(count_per_turn)) * PI / 180 * 360;
    return pos_in_radian;
    
}

int ESCON_encoder::getRevolutions(void) {
    revolutions = floor( double(pulses / (2 * double(count_per_turn))) );
    return revolutions;
}


void ESCON_encoder::encode(void) {

    int chanA  = channelA.read();
    int chanB  = channelB.read();

    //2-bit state.
    currState = (chanA << 1) | (chanB);

    //11->00->11->00 is counter clockwise rotation or "forward".
    if ((prevState == 0x3 && currState == 0x0) ||
             (prevState == 0x0 && currState == 0x3)) {

        pulses++;

    }
    //10->01->10->01 is clockwise rotation or "backward".
    else if ((prevState == 0x2 && currState == 0x1) ||
              (prevState == 0x1 && currState == 0x2)) {
                  
        pulses--;
        
    }
    prevState = currState;

}