/** ISL MBED Libraries
 * Copyright (C) 2019 - Invisible Systems Ltd. All Rights Reserved.
 *
 * <b>         M24512.h</b>
 * <p>         M24512 EEPROM Driver </p>
 * @version    1.0.1
 * @since      06 June 2019
 * @author     Martyn Gilbertson
 * <p>
 *  v1.0.0   - 06 June 2019
 *          [+] Initial release
 *
 *  v1.0.1   - 16 September 2019
 *          [+] Add retry to Write()
 *
 * *  v1.0.2   - 16 October 2019
 *          [*] Decreased retry limit as to not WDT
 *  		[+] Allow ready routine from public
 *
 *  v1.0.2   - 16 October 2019
 *          [*] Decreased retry limit as to not WDT
 *          [+] Allow ready routine from public
 *
 *  v1.0.3   -  13 February 2020
 *          [+] Improved Timeout on read
 *          [+] Decreased default i2c frequency for stability
 */
#ifndef _M24512_H_
#define _M24512_H_

#include <stdint.h>
#include <i2c.h>

/** Driver for reading and writing to M24512 eeprom
 *
 * Example:
 * @code
 * // Write "test" to page 0 and read back
 * #include "M24512.h"
 *
 * M24512 eeprom(I2C_SDA, I2C_SCL);
 *
 * int main() {
 * 	M24512::status_t ret;
 * 	char data[32] = {'t','e','s','t'};
 *
 *	ret = eeprom.write(0, (char*)data, 4);
 *	if ( ret != M24512::M24512_SUCCESS )
 *	{
 *		printf("Error writing this! %2.2X\n", ret );
 *	}
 *
 *	memset( data, 0, sizeof(data) );
 *	ret = eeprom.read(0, (char*)&data, 4);
 *	if ( ret !=  M24512::M24512_SUCCESS )
 *	{
 *		printf("Error reading %2.2X!\n", ret );
 *	}
 *	printf("Read: %s\n", data );
 *  return 0;
 * }
 * @endcode
 */
class M24512
{

	public:

		 /** MBED uses 8 bit addressing so shift to 7bit
		  *   Default value 0x50 is with address pins hi-z
		  */
		 #define M24512_DEFAULT_ADDRESS (uint8_t)(0x50 << 1)


		/** Return values for each function
	 	*/
		typedef enum {
            /*! Succes */
			M24512_SUCCESS		= 0,
            /*! I2C Ack error */
			M24512_I2C_ERROR    = 1,
            /*! Device Ready timed out */
			M24512_RDY_TIMEOUT  = 2,
            /*! Memory overflow */
			M24512_MEM_OVERFLOW = 3
		} status_t;


		/** Initialise M24512 Interface
		 *
		 * @param sda  - I2C Data pin
		 * @param scl  - I2C Clock pin
		 * @param addr - Hardware address of device (defaults to 0x50)
		 */
		M24512(PinName sda, PinName scl, uint8_t address = M24512_DEFAULT_ADDRESS);


		/** Destructor
		 */
		~M24512(void);


		/** Read continuous from address
		 *
		 * @param addr  - address to start reading from
		 * @param data  - pointer to data buffer (must be allocated)
		 * @param size  - size of data to read back into data pointer
		 */
		status_t read(uint16_t addr,  char* data,  uint16_t size);


		/** Write continuous to address
		 *
		 * @param addr  - address to start writing at
		 * @param data  - pointer to data buffer must be allocated
		 * @param size  - size of data to write
		 */
		status_t write(uint16_t addr, const char* data, uint16_t size);

		/** Set the i2c frequency
		 *
		 * @param hz  - frequency in hertz
		 */
		void frequency(uint32_t hz);


		/** Get page size
		 */
		uint16_t get_page_size() const;


		/** Get page count
		 */
		uint16_t get_page_count() const;


		/** Get memory size in KiB
		 */
		uint32_t get_total_memory() const;


        /** Check status of Memory chip after writing data
         *
         * @returns  -  @see status_t enum typedef
         */
		status_t ready(void);

	private:

        /*! i2c driver class */
		mbed::I2C _i2c;

        /*! memory address */
		uint8_t _addr;

        /** Write to single page
         * @note  this will only write a maximum of %page_size%
         *
         * @param addr  - address to start writing at
         * @param data  - pointer to data buffer must be allocated
         * @param size  - size of data to write maximum %page_size%
         */
		status_t write_page(uint16_t addr, const char* data,  uint16_t size);


};

#endif /* _M24512_H_ */

