#include "mbed.h"

#if defined(FEATURE_COMMON_PAL)
#include "mbed_trace.h"
#define TRACE_GROUP "APPL"
#else
#define tr_debug(...) (void(0)) //dummies if feature common pal is not added
#define tr_info(...)  (void(0)) //dummies if feature common pal is not added
#define tr_error(...) (void(0)) //dummies if feature common pal is not added
#endif //defined(FEATURE_COMMON_PAL)

#define TW_TCP_TEST_DEVICE_NO 1
#define TW_TCP_TEST_SERVER_IP "1.221.6.243"
#define TW_TCP_TEST_SERVER_PORT 41112

DigitalOut led(PA_5);

#include "ESP8266Interface.h"
ESP8266Interface wifi(PC_4, PC_5);

#include "DHT.h"
DHT sensor(PC_2, DHT11); 

const char *sec2str(nsapi_security_t sec)
{
    switch (sec) {
        case NSAPI_SECURITY_NONE:
            return "None";
        case NSAPI_SECURITY_WEP:
            return "WEP";
        case NSAPI_SECURITY_WPA:
            return "WPA";
        case NSAPI_SECURITY_WPA2:
            return "WPA2";
        case NSAPI_SECURITY_WPA_WPA2:
            return "WPA/WPA2";
        case NSAPI_SECURITY_UNKNOWN:
        default:
            return "Unknown";
    }
}

void scan_demo(WiFiInterface *wifi)
{
    WiFiAccessPoint *ap;
 
    tr_info("Scan:\r\n");
 
    int count = wifi->scan(NULL,0);
 
    /* Limit number of network arbitrary to 15 */
    count = count < 15 ? count : 15;
 
    ap = new WiFiAccessPoint[count];
    count = wifi->scan(ap, count);
    for (int i = 0; i < count; i++)
    {
        tr_info("Network: %s secured: %s BSSID: %hhX:%hhX:%hhX:%hhx:%hhx:%hhx RSSI: %hhd Ch: %hhd\r\n", ap[i].get_ssid(),
               sec2str(ap[i].get_security()), ap[i].get_bssid()[0], ap[i].get_bssid()[1], ap[i].get_bssid()[2],
               ap[i].get_bssid()[3], ap[i].get_bssid()[4], ap[i].get_bssid()[5], ap[i].get_rssi(), ap[i].get_channel());
    }
    tr_info("%d networks available.\r\n", count);
 
    delete[] ap;
}

int main()
{
    TCPSocket tcp_socket;

    printf("\r\nSSU Temperature/Humidity WiFi Comm Board\r\n");
    mbed_trace_init();
    
    int ret;
    tr_info("WiFi Connecting...");
    while(wifi.connect(MBED_CONF_APP_WIFI_SSID, MBED_CONF_APP_WIFI_PASSWORD, NSAPI_SECURITY_WPA_WPA2) != 0) {
        tr_err("WiFi Connection error, Retry");
    }
    
    tr_info("Successfully connected to WiFi AP");
    tr_debug("MAC: %s", wifi.get_mac_address());
    tr_debug("IP: %s", wifi.get_ip_address());
    tr_debug("Netmask: %s", wifi.get_netmask());
    tr_debug("Gateway: %s", wifi.get_gateway());
    tr_debug("RSSI: %d", wifi.get_rssi());
    
    tr_info("Create WiFi TCP Socket");
    if((ret = tcp_socket.open(&wifi)) < 0) {
        tr_err("Error: Cannot Open UDP Socket (err code:%d)", ret);
        NVIC_SystemReset();
    }
    tr_info("Successfully created WiFi TCP Socket");
    
    while(1) {
        int err;
        float humid = 0.0f;
        float temp = 0.0f;
        
        char json_buf[100];
        
        //Read Humid, Temp
        err = sensor.readData();
        if (err == 0) {
            humid = sensor.ReadHumidity();
            temp = sensor.ReadTemperature(CELCIUS);
            
            tr_info("Humidity = %4.2f%% Temperature = %4.2fC", humid, temp);
            sprintf(json_buf, "{\r\n\t\"id\":%d,\r\n\t\"temp\":%f,\r\n\t\"humid\":%f\r\n}\r\n", TW_TCP_TEST_DEVICE_NO, temp, humid);
            
            if((ret = tcp_socket.connect(TW_TCP_TEST_SERVER_IP, TW_TCP_TEST_SERVER_PORT)) < 0) {
                tr_err("Cannot connect to TCP Server");
            } else {
                if((ret = tcp_socket.send(json_buf, strlen(json_buf))) < 0) {
                    tr_err("Cannot send to TCP Server");
                } else {
                    tr_info("Sucessfully send TCP packet via WiFi");
                }
            }
        } else {
            tr_err("Cannot Read Humid & Temp");
        }

        led = !led;
        wait(5);
    }
}