/* Demo for mbed Library for FTDI FT800  Enbedded Video Engine "EVE"
 * based on Original Code Sample from FTDI
 * ported to mbed by Peter Drescher, DC2PD 2014
 * Released under the MIT License: http://mbed.org/license/mit */

#include "mbed.h"
#include "FT_Platform.h"

DigitalOut myled(LED1);
FT800 TFT(p11,p12,p13,p14,p15,p16);       // the FT800 is connected to SPI 11-13

/* API to demonstrate calibrate widget/functionality */
ft_void_t CoPro_Widget_Calibrate()
{
    /*************************************************************************/
    /* Below code demonstrates the usage of calibrate function. Calibrate    */
    /* function will wait untill user presses all the three dots. Only way to*/
    /* come out of this api is to reset the coprocessor bit.                 */
    /*************************************************************************/
    {

        TFT.Ft_Gpu_CoCmd_Dlstart();                                 // start a new display command list
        TFT.Ft_App_WrCoCmd_Buffer(CLEAR_COLOR_RGB(64,64,64));       // set the clear color R, G, B
        TFT.Ft_App_WrCoCmd_Buffer(CLEAR(1,1,1));                    // clear buffers -> color buffer,stencil buffer, tag buffer
        TFT.Ft_App_WrCoCmd_Buffer(COLOR_RGB(0xff,0xff,0xff));       // set the current color R, G, B
        TFT.Ft_Gpu_CoCmd_Text((TFT.FT_DispWidth/2), (TFT.FT_DispHeight/2), 27, OPT_CENTER, "Please Tap on the dot");  // draw Text at x,y, font 27, centered
        TFT.Ft_Gpu_CoCmd_Calibrate(0);                              // start the calibration of touch screen
        TFT.Ft_App_Flush_Co_Buffer();                               // download the commands into FT800 FIFO
        TFT.Ft_Gpu_Hal_WaitCmdfifo_empty();                         // Wait till coprocessor completes the operation
    }
}


/***************************************************************************/
/* Show a Screen with Text for 5 seconds                                   */
/* A spinner shows the delay                                               */
/***************************************************************************/

ft_void_t API_Screen(ft_char8_t *str)
{
    TFT.Ft_Gpu_CoCmd_Dlstart();                                   // start a new display command list
    TFT.Ft_App_WrCoCmd_Buffer(CLEAR_COLOR_RGB(255,255,255));      // set the clear color to white
    TFT.Ft_App_WrCoCmd_Buffer(CLEAR(1,1,1));                      // clear buffers -> color buffer,stencil buffer, tag buffer

    TFT.Ft_App_WrCoCmd_Buffer(COLOR_RGB(0x80,0x80,0x00));         // set current color
    TFT.Ft_Gpu_CoCmd_Text((TFT.FT_DispWidth/2), TFT.FT_DispHeight/2, 31, OPT_CENTERX, str); // draw Text with font 31
    TFT.Ft_App_WrCoCmd_Buffer(COLOR_RGB(0xFF,0x00,0x00));         // change current color
    TFT.Ft_Gpu_CoCmd_Spinner((TFT.FT_DispWidth/2),TFT.FT_DispHeight/4, 0,0); // draw a animated spinner

    TFT.Ft_App_WrCoCmd_Buffer(DISPLAY());                         // Display the image
    TFT.Ft_Gpu_CoCmd_Swap();                                      // Swap the current display list
    TFT.Ft_App_Flush_Co_Buffer();                                 // Download the command list into fifo

    TFT.Ft_Gpu_Hal_WaitCmdfifo_empty();                           // Wait till coprocessor completes the operation
    TFT.Ft_Gpu_Hal_Sleep(5000);                                   // Wait 5 s
}



int main()
{

    /*************************************************************************/
    /* Below code demonstrates the usage of track function. Track function   */
    /* tracks the pen touch on any specific object. Track function supports  */
    /* rotary and horizontal/vertical tracks. Rotary is given by rotation    */
    /* angle and horizontal/vertucal track is offset position.               */
    /*************************************************************************/
    {
        ft_uint32_t TrackRegisterVal = 0;
        ft_uint16_t angleval = 0,slideval = 0,scrollval = 0;

        API_Screen("DEMO  START");                  // Show start screen
        CoPro_Widget_Calibrate();                   // calibrate the touch screen

        /* Set the tracker for 3 objects */
        TFT.Ft_Gpu_CoCmd_Track(TFT.FT_DispWidth/2, TFT.FT_DispHeight/2, 1,1, 10);
        TFT.Ft_Gpu_CoCmd_Track(40, (TFT.FT_DispHeight - 40), (TFT.FT_DispWidth - 80),8, 11);
        TFT.Ft_Gpu_CoCmd_Track((TFT.FT_DispWidth - 40), 40, 8,(TFT.FT_DispHeight - 80), 12);
        /* Download the commands into fifo */
        TFT.Ft_App_Flush_Co_Buffer();

        /* Wait till coprocessor completes the operation */
        TFT.Ft_Gpu_Hal_WaitCmdfifo_empty();

        /* update the background color continuously for the color change in any of the trackers */
        /* the demo is updating the screen in a endless loop                                    */
        while(1) {
            ft_uint8_t tagval = 0;
            TrackRegisterVal = TFT.Ft_Gpu_Hal_Rd32(REG_TRACKER);    // check if one of the tree objectes is touched
            tagval = TrackRegisterVal & 0xff;
            if(0 != tagval) {
                if(10 == tagval) {                                  // the Rotary dial is touched
                    angleval = TrackRegisterVal>>16;                // get the new angle
                } else if(11 == tagval) {                           // the vertical slider is touched
                    slideval = TrackRegisterVal>>16;                // get new slider value
                } else if(12 == tagval) {                           // the horizontal slider is touched
                    scrollval = TrackRegisterVal>>16;               // get new slider value
                    if((scrollval + 65535/10) > (9*65535/10)) {
                        scrollval = (8*65535/10);
                    } else if(scrollval < (1*65535/10)) {
                        scrollval = 0;
                    } else {
                        scrollval -= (1*65535/10);
                    }
                }
            }
            /* Display a rotary dial, horizontal slider and vertical scroll */

            TFT.Ft_App_WrCoCmd_Buffer(CMD_DLSTART);                     // start a new display command list

            {
                // calculate the new background color based on the sliders and
                ft_int32_t tmpval0,tmpval1,tmpval2;                     // the rotary
                ft_uint8_t angval,sldval,scrlval;

                tmpval0 = (ft_int32_t)angleval*255/65536;
                tmpval1 = (ft_int32_t)slideval*255/65536;
                tmpval2 = (ft_int32_t)scrollval*255/65536;

                angval = tmpval0&0xff;
                sldval = tmpval1&0xff;
                scrlval = tmpval2&0xff;
                TFT.Ft_App_WrCoCmd_Buffer(CLEAR_COLOR_RGB(angval,sldval,scrlval)); // set the new clear color
            }
            TFT.Ft_App_WrCoCmd_Buffer(CLEAR(1,1,1));                               // clear buffers -> color buffer,stencil buffer, tag buffer
            TFT.Ft_App_WrCoCmd_Buffer(COLOR_RGB(0xff,0xff,0xff));                  // set current color to white

            /* Draw dial with 3d effect */
            TFT.Ft_Gpu_CoCmd_FgColor(0x00ff00);                                    // forground color green
            TFT.Ft_Gpu_CoCmd_BgColor(0x800000);                                    // background color
            TFT.Ft_App_WrCoCmd_Buffer(TAG(10));                                    // assign TAG value 10
            TFT.Ft_Gpu_CoCmd_Dial((TFT.FT_DispWidth/2), (TFT.FT_DispHeight/2), (TFT.FT_DispWidth/8), 0, angleval); // draw dial at display center, dial point at angleval

            /* Draw slider with 3d effect */
            TFT.Ft_Gpu_CoCmd_FgColor(0x00a000);                                     // forground color green
            TFT.Ft_Gpu_CoCmd_BgColor(0x800000);                                     // background color
            TFT.Ft_App_WrCoCmd_Buffer(TAG(11));                                     // assign TAG value 11
            TFT.Ft_Gpu_CoCmd_Slider(40, (TFT.FT_DispHeight - 40), (TFT.FT_DispWidth - 80),8, 0, slideval, 65535);  // draw Slider, position slideval, max 65535

            /* Draw scroll with 3d effect */
            TFT.Ft_Gpu_CoCmd_FgColor(0x00a000);                                     // forground color green
            TFT.Ft_Gpu_CoCmd_BgColor(0x000080);                                     // background color
            TFT.Ft_App_WrCoCmd_Buffer(TAG(12));                                     // assign TAG value 11
            TFT.Ft_Gpu_CoCmd_Scrollbar((TFT.FT_DispWidth - 40), 40, 8, (TFT.FT_DispHeight - 80), 0, scrollval, (65535*0.2), 65535); // draw scrollbar , position scrollval, max 65535

            TFT.Ft_Gpu_CoCmd_FgColor(TAG_MASK(0));                                  // reset TAG_MASK
            TFT.Ft_App_WrCoCmd_Buffer(COLOR_RGB(0xff,0xff,0xff));                   // change current color to white
            TFT.Ft_Gpu_CoCmd_Text((TFT.FT_DispWidth/2), ((TFT.FT_DispHeight/2) + (TFT.FT_DispWidth/8) + 8), 26, OPT_CENTER, "Rotary track"); // display text
            TFT.Ft_Gpu_CoCmd_Text(((TFT.FT_DispWidth/2)), ((TFT.FT_DispHeight - 40) + 8 + 8), 26, OPT_CENTER, "Horizontal track");           // display text
            TFT.Ft_Gpu_CoCmd_Text((TFT.FT_DispWidth - 40), 20, 26, OPT_CENTER, "Vertical track");                                            // display text

            TFT.Ft_App_WrCoCmd_Buffer(DISPLAY());                                   // Display the image
            TFT.Ft_Gpu_CoCmd_Swap();                                                // Swap the current display list

            TFT.Ft_App_Flush_Co_Buffer();                                           // Download the commands into fifo

            TFT.Ft_Gpu_Hal_WaitCmdfifo_empty();                                     // Wait till coprocessor completes the operation

            TFT.Ft_Gpu_Hal_Sleep(10);                                               // wait 10ms
        }  // end of display loop


    }

}
