#include "mbed.h"
#include "EthernetInterface.h"
#include "HTTPServer.h"
#include "mbed_rpc.h"
#include "RPCFunction.h"
#include "AFE_4404.h"

EthernetInterface eth;  
HTTPServer svr;

int baudRate = 115200;
Serial pc(USBTX, USBRX); 

extern int32_t data;
extern bool dataAvailable;
// p22: rxSupplyEn, p8: txSupplyEn, p6: resetz, p7: powerEn,
// p5: drdy, p21: clk, p9: sda, p10: scl   
AFE_4404 afe(p22, p8, p6, p7, p5, p21, p9, p10);

Thread rpcThread;
//automatically lock rpcBusy when it's been used
volatile bool rpcBusy = true;
void afeRpc(Arguments *input, Reply *output);
RPCFunction afeRpcFunction(&afeRpc, "afe");

void afeRpc(Arguments *input, Reply *output){
    
    int32_t registerAddress;
    int32_t registerData;
    int32_t suspend; // prevents main from writing to data if it's 1 
   
    // prevent main loop from writing data to serial
    rpcBusy = true;
    Thread::wait(100);
    
    registerAddress = input->getArg<int32_t>();
    registerData = input->getArg<int32_t>();
    suspend = input->getArg<int32_t>();
    
    // write data to afe register
    afe.writeData(
        static_cast<uint8_t>(registerAddress), 
        static_cast<uint32_t>(registerData));
    
    output->putData(registerAddress);
    output->putData(registerData);
    
    // suspend serial writes in main loop. functionality isn't currently
    // being used, so rpcBusy is always false in current C# GUI version
    rpcBusy = (suspend == 1) ? true : false;
}

void rpcThreadRoutine() {
    
    printf("Setting up...\n");
    eth.init();
    
    // wait for a while because the initialization doesn't work without the delay
    // maybe the initialization is done asynchronously??
    wait(10.0);
    
    int ethErr = eth.connect();
    
    if(ethErr < 0) 
        printf("Error %d in setup.\n", ethErr);
  
    svr.addHandler<HTTPRpcRequestHandler>("/rpc");
    //attach server to port 80
    svr.start(80, &eth);
    
    // gives enough time to pause terminal application to obtain IP address
    Thread::wait(2000);  
    printf("Listening...\n");
    
    rpcBusy = false;
    
    //Listen indefinitely
    while(true) {
        svr.poll();
        Thread::wait(1000);
    }    
}

int main() {
    
    pc.baud(baudRate);
    
    rpcThread.start(rpcThreadRoutine);   
    // spin until RPC is setup
    while(rpcBusy) {}
    
    afe.powerUpSequence();
    
    while(true){
        if(dataAvailable && !rpcBusy) {
            pc.printf("%d ", data);  
            dataAvailable = false;
        } 
    }
}