#include "DtPWM.h"

DtPWM::DtPWM(bool enabled){
    
    LEDs = false; //LEDs are off on default
    
    LPC_SC->PCONP|=(1<<6);                          //enable power on PWM1
    LPC_SC->PCLKSEL0|=(1<<12);                      //run PWM1 clock at prescaler 1 (96MHz)
    
    
    LPC_PWM1->MCR|=(1<<1);   //no interrupt, no reset, no stop at TC compare match
        
    
    LPC_PWM1->CTCR|=(0<<0);                         //timer mode
    LPC_PWM1->PCR|= (1<<4) | (1<<2);     //double edge control on channel 2 and 4
    
    LPC_PWM1->PR=0;                                 // don't pre scale PWM so its 48 MHz
    
    setDtPWM(1000, 0, 0);
    
    LPC_PWM1->TCR|= (1<<3);     //enable PWM timer mode
    
    if (enabled) {
        enable(); //enable output
    }
    else {
        disable();
    }
}



void DtPWM::setDtPWM(double p_us, float d, double dt_us) {
    
    p_tcks = 96 * p_us - 1;
    dt_tcks = 96 * dt_us;
       
    d_max = (p_us - 2 * dt_us) / p_us;
    
    setD( d );
    
    LPC_PWM1->MR0=p_tcks;
    LPC_PWM1->LER|=(1<<0);
    
    
    
}

void DtPWM::setGPIO(char channel, bool high) {
    
    // set GPIO on these pins to low
    
    uint8_t mask = 0; 
    if( channel == 'h' || channel == 'b' || channel == 'a' ) 
        mask |= (1 << 1);  // P2.1
    if( channel == 'l' || channel == 'b' || channel == 'a' ) 
        mask |= (1 << 3);  // P2.3
    if( channel == 't' || channel == 'a')
        mask |= (1 << 4);  // P2.4
        
    LPC_GPIO2->FIODIR0 |= mask;     // set pins as outputs
    LPC_GPIO2->FIOMASK0 &= ~mask;    // clear the mask bits
    
    if( high ) {
        LPC_GPIO2->FIOSET0 |= mask; // set pins to high
    } else {
        LPC_GPIO2->FIOCLR0 |= mask; // set pins to low
    }
    
}

void DtPWM::setPINS(char channel, uint32_t function) {
    uint32_t mask = 0;
    if( channel == 'h' || channel == 'b' || channel == 'a' ) 
        mask |= (0x3 << 2);  // P2.1
    if( channel == 'l' || channel == 'b' || channel == 'a' ) 
        mask |= (0x3 << 6);  // P2.3
    if( channel == 't' || channel == 'a')
        mask |= (0x3 << 8);  // P2.4
        
    LPC_PINCON->PINSEL4 = (LPC_PINCON->PINSEL4 & ~mask) | (function & mask);
    
    // set pins to 10, no pull up / pull down
    LPC_PINCON->PINMODE4 = (LPC_PINCON->PINMODE4 & ~mask) | (FUNC_10 & mask);
}

void DtPWM::enable(char channel) {
    
    // set pins to PWM function (01)
    setPINS(channel, FUNC_01);
    
    // turn high side channel on (PWM1.2)
    if( channel == 'h' || channel == 'b' || channel == 'a' ) {        
        LPC_PWM1->PCR |= (1<<10);       // enable PWM output 2     
    }    
    // turn low side channel on (PWM1.4)
    if( channel == 'l' || channel == 'b' || channel == 'a' ) {        
        LPC_PWM1->PCR |= (1<<12);       // enable PWM output 2       
    }
    // turn trigger channel on (PWM1.5);
    if( channel == 't' || channel == 'a') {
        LPC_PWM1->PCR |= (1<<13);    // enable PWM output 5
    }
        
    LPC_PWM1->TCR|=(1<<0);  //enable counter
}

void DtPWM::setOutput(char channel, bool high) {
    setPINS( channel, FUNC_00 );
    setGPIO( channel, high );
}
void DtPWM::setLEDs( bool on  ) {
    
    uint32_t mask = (0x3<<14) | (0x3<<8);
    
    if( on && !LEDs ) {        
    
        LEDs_pinsel = LPC_PINCON->PINSEL3;
        
        /* set ports P1.20 to PWM1.2 and P1.23 to PWM1.2 (LED2 & 4) */
        LPC_PINCON->PINSEL3 &= ~mask; // reset bits 
        LPC_PINCON->PINSEL3 |= (0x2<<14) | (0x2<<8);      // set functions 10 (PWM)
                
        LEDs_pinmode = LPC_PINCON->PINMODE3;
        LPC_PINCON->PINMODE3 &= ~mask; // reset bits
        LPC_PINCON->PINMODE3 |= (2<<14) | (2<<8);        // no pull ups/pull downs
        
        LEDs = true;
    }   
    if ( !on && LEDs ) {
        /* reset pinsel and pinmode registers */
        LPC_PINCON->PINSEL3 &= ~mask; // reset bits
        LPC_PINCON->PINSEL3 |= LEDs_pinsel & mask;
        
        LPC_PINCON->PINMODE3 &= ~mask; // reset bits
        LPC_PINCON->PINMODE3 |= LEDs_pinmode & mask;
        
        LEDs = false;
    }
}

int DtPWM::isEnabled(char channel) {    
    
    if( channel == 'h' ) {
        return LPC_PWM1->PCR & (1<<10);       // check PWM output 2     
    }
    else if( channel == 'l' ) {
        return LPC_PWM1->PCR & (1<<12);       // check PWM output 4    
    }
    else if( channel == 'b' ) {
        return LPC_PWM1->PCR & ( (1<<12) | (1<<10) );  // check PWM output 2 and 4  
    }
    else if( channel == 't' ) {
        return LPC_PWM1->PCR & (1<<13);       // check PWM output 5  
    }
    else if( channel == 'a' ) {
        return LPC_PWM1->PCR & (0x3F<<9);     // return all enable bits
    }
    
    return -1;
}

void DtPWM::disable(char channel) {    
    
    
    // turn high side channel off (PWM1.2)
    if( channel == 'h' || channel == 'b' || channel == 'a' ) {        
        LPC_PWM1->PCR &= ~(1<<10);       // disable PWM output 2     
    }    
    // turn low side channel off (PWM1.4)
    if( channel == 'l' || channel == 'b' || channel == 'a' ) {        
        LPC_PWM1->PCR &= ~(1<<12);       // disable PWM output 2       
    }
    // turn trigger channel off (PWM1.5);
    if( channel == 't' || channel == 'a') {
        LPC_PWM1->PCR &= ~(1<<13);    // disable PWM output 5
    } 
    
    setGPIO( channel, 0 ); // set the gpio to low
    setPINS( channel, FUNC_00 );
    
}

void DtPWM::setInterrupts(uint32_t flags, int prio){
    if (flags == 0) {
        /* interrupt disable */
        NVIC_DisableIRQ(PWM1_IRQn);
        LPC_PWM1->MCR &= ~(DTPWM_SIA);
    } else {
        /* interrupt enable */
        LPC_PWM1->MCR &= ~(DTPWM_SIA);
        LPC_PWM1->MCR |= flags;        
        NVIC_EnableIRQ(PWM1_IRQn);
    }
    
    if( prio >=0 && prio <=32 ) {
        NVIC_SetPriority(PWM1_IRQn, prio);  //set priority of these interrupts
    }
    
}

void DtPWM::setD(float d){
    unsigned int d_tcks = d * p_tcks;
     
    LPC_PWM1->MR1=0;                // high side turns on immediately                                           
    LPC_PWM1->MR2=d_tcks;           // high side turns off at the end of d
    
    if( d < d_max ) {
        LPC_PWM1->MR3=d_tcks + dt_tcks;
    }
    else {
        LPC_PWM1->MR3 = p_tcks + 1; // never reached 
    }                                                               
                             
    LPC_PWM1->MR4 = p_tcks - dt_tcks;  
    
    LPC_PWM1->MR5 = (p_tcks + d_tcks) / 2;   // in the middle of when high side is not on
        
    // special occasion when d is negative
    if ( d < 0 ) {
        LPC_PWM1->MR1 = p_tcks + 1;
        LPC_PWM1->MR2 = p_tcks + 1;
        d_tcks = -d * p_tcks / 2;
        
        if ( d_tcks < dt_tcks ) {
            LPC_PWM1->MR3 = dt_tcks - d_tcks;
            LPC_PWM1->MR4 = p_tcks - dt_tcks + d_tcks;
        }
        else {
            LPC_PWM1->MR3 = 0;
            LPC_PWM1->MR4 = p_tcks + 1;
        }  
          
    }  
    
    LPC_PWM1->LER|=(1<<5) | (1<<4) | (1<<3) | (1<<2) | (1<<1);             
           
}