#include "mbed.h"

#ifndef DTPWM_H
#define DTPWM_H

/** 
* Library for fast and high resolution opposite PWM output with 
* controllable dead times
* Version 3
*/

/** definitions for interrupt flags, usage in IRS:
* if ( LPC_PWM1->IR & DTPWM_IF0 ) {
*     // do something on MR0 interrupt
*     LPC_PWM1->IR |= DTPWM_IF0; // reset interrupt flag  
* }
*/
#define DTPWM_IF0 (1 << 0U)
#define DTPWM_IF1 (1 << 1U)
#define DTPWM_IF2 (1 << 2U)
#define DTPWM_IF3 (1 << 3U)
#define DTPWM_IF4 (1 << 8U)
#define DTPWM_IF5 (1 << 9U)
#define DTPWM_IF6 (1 << 10U)

/** definitions for interrupt enable flags, usage:
* pwm.setInterrupts(DTPWM_SI0 | DTPWM_SI5); //to set MR0 & MR5 interrupts
*/
#define DTPWM_SI0 (1 << 0U)
#define DTPWM_SI1 (1 << 3U)
#define DTPWM_SI2 (1 << 6U)
#define DTPWM_SI3 (1 << 9U)
#define DTPWM_SI4 (1 << 12U)
#define DTPWM_SI5 (1 << 15U)
#define DTPWM_SI6 (1 << 18U)

#define DTPWM_SIA (DTPWM_SI0 | DTPWM_SI1 | DTPWM_SI2 | DTPWM_SI3 | DTPWM_SI4 | DTPWM_SI5 | DTPWM_SI6)

#define FUNC_00 0x00000000
#define FUNC_01 0x55555555
#define FUNC_10 0xAAAAAAAA
#define FUNC_11 0xFFFFFFFF

class DtPWM{
public:
    /**
    * Create a dtPWM object connected to channel 2 and 4, optionally to LED 2&4
    *
    * @param ebaled - start pwm enabled, output as soon as object is created (default)
    */
    DtPWM(bool enable = true);
    
    /**
    * enable pin outputs and start the timer
    *
    * @channel - determines which channels are enabled. 
    *               'h' for high (PWM1.2)
    *               'l' for low (PWM1.4)
    *               'b' for both (PWM1.2 & 1.4) (default)
    *               't' for trigger channel (PWM1.5)
    *               'a' for all three 
    */
    void enable(char channel = 'b');
    
    /**
    * check if pins are enabled
    *
    * @channel - determines which channels are checked. 
    */
    int isEnabled(char channel = 'b');
        
    /**
    * disable pin outputs 
    *
    * @channel - determines which channels are disabled. 
    */
    void disable(char channel = 'b'); 
    
    
    
    /**
    * Set the period, duty cycle, and dead time.
    *
    * @param p_us - period in micro seconds
    * @param d - duty cycle from 0-1 as float
    * @param dt_us - dead time between switching in micro seconds
    */
    void setDtPWM(double p_us, float d, double dt_us);
    
    /**
    * Set which interrupts are enabled
    *
    * @param flags - OR'd bits for the interrupt register, 0 to disable interrupts
    *   use the definitions eg. pwm.setInterrupts(DTPWM_SI0 | DTPWM_SI5); to set MR0 & MR5
    * @param priority - set interrupt priority level. -1 (default) leaves them unchanged
    */
    void setInterrupts(uint32_t flags, int prio = -1);
    
    /**
    * Value for maximum duty cycle that fulfils minimum dead times 
    */
    float d_max;
    
    /**
    * Function to just set the duty cycle.
    *
    * @param d - duty cycle from 0-1 as float no checking is done in this function
    */
    void setD(float d);
    
    /**
    * enable or disable LED2 & LED4 to be tied to PWM1.2 & PWM1.4
    *
    * @on - true (default) to enable LEDs, false to disable LEDs
    * 
    * Restorse pin configuration to before the enable. Enabling when they are
    * already enabled or disabling when they are disabled does nothing.
    */
    void setLEDs( bool on = true );
    
    /**
    * Force outputs to certain value
    *
    * @channel - determines which channels are affected. 
    * @high - the value to be set (high or low)
    */
    void setOutput(char channel, bool high);
    
private:
    unsigned int p_tcks;        // clock ticks in period
    unsigned int dt_tcks;       // dead time ticks
    
    bool LEDs;                  // flag for LED output
    uint32_t LEDs_pinsel;       // stores the PINSEL3 register when LEDs are enabled
    uint32_t LEDs_pinmode;      // stores the PINMODE3 register when LEDs are enabled
    
    /** 
    * Sets the GPIO value for the selected channels.
    *
    * @channel - select which channels are affected
    * @high - if true, set them to high, if false (default) to low
    */
    void setGPIO(char channel, bool high = false);
    
    /** 
    * Sets the function of the pins selected by channel.
    *
    * @channel - select which channels are affected
    * @function - FUNC_00 for GPIO, FUNC_01 for PWM
    */
    void setPINS(char channel, uint32_t function);
    
};
#endif