
/*
Copyright (c) 2010 Donatien Garnier (donatiengar [at] gmail [dot] com)
 
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:
 
The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.
 
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "netCfg.h"
#if NET_ZG2100

#include "zg_defs.h"
#include "zg_net.h"
#include "zg_drv.h"

#include "lwip/opt.h"

#include "lwip/def.h"
#include "lwip/pbuf.h"
#include "lwip/sys.h"
#include "lwip/stats.h"
#include "netif/etharp.h"

#include "string.h"

#ifdef __LWIP_DEBUG
#undef __LWIP_DEBUG
#endif
//#define __DEBUG
#include "dbg/dbg.h"

#define IFNAME0 'w'
#define IFNAME1 'l'

//void hexdump(byte* buffer, int size);
#define hexdump(x,y)

const char snap[ZG_SNAP_LEN] = { 0xAA, 0xAA, 0x03, 0x00, 0x00, 0x00 }; //Snap word, see IEEE 802-2001 10.3 Subnetwork Access Protocol (p. 23 ~ 29)
struct netif* zg_netif;

//To be called by Lwip
static err_t zg_output(struct netif *netif, struct pbuf *p) //Transfer an ethernet frame : convert into ZG frame & TX it
{
  DBG("\r\nOut packet\r\n");
  
  while( zg_send_is_busy() )
  {
    zg_process(); //Ugly but prevents lwip from feeding other packets
  }
  #if ETH_PAD_SIZE
    pbuf_header(p, -ETH_PAD_SIZE); //Drop padding word
  #endif
  
  ZG_ETH_HDR* eth_hdr = (ZG_ETH_HDR*) p->payload;
  
  ZG_TX_HDR* tx_hdr = (ZG_TX_HDR*)fifo_buf;
  memset((void*)tx_hdr, 0, sizeof(ZG_TX_HDR));
  
  //Convert headers properly
  tx_hdr->zero = HTODS( 0 );
  memcpy( tx_hdr->dest, eth_hdr->dest, ZG_MACADDR_LEN );
  memcpy( tx_hdr->snap, snap, ZG_SNAP_LEN );
  tx_hdr->type = eth_hdr->type; //Lwip has already put the type in BE
  
  DBG("\r\nEth header\r\n");
  hexdump((byte*)eth_hdr, sizeof(ZG_ETH_HDR));
  
  DBG("\r\nZg header\r\n");
  hexdump((byte*)tx_hdr, sizeof(ZG_TX_HDR));
  
  //Open TX fifo & send header
  zg_send_start();
  zg_send( (byte*)tx_hdr, sizeof(ZG_TX_HDR) );
  
  //Send remaining payload of this buf
  zg_send( (byte*)(((byte*)p->payload) + sizeof(ZG_ETH_HDR)), (p->len - sizeof(ZG_ETH_HDR)) );
  
  while(p->next != NULL)
  {
    p = p->next;
    zg_send((byte*)p->payload, p->len);
    //FIX: Watchout, p content MUST be preserved for proper TCP functionality!!!  
  }

  zg_send_end();

  #if ETH_PAD_SIZE
    pbuf_header(p, ETH_PAD_SIZE); //Reclaim padding word
  #endif
  
  LINK_STATS_INC(link.xmit);
  return ERR_OK;  
}

//Callback from zg_drv
void zg_on_input(byte* buf, int len) //On reception of a ZG frame : convert into Eth frame & feed lwip
{
  struct pbuf *frame, *p; //Lwip buffers
  byte* eth_buf; //Position of the Ethernet packet buffer
  int eth_len; //Length of this buffer
  ZG_ETH_HDR eth_hdr_data; //Temporary buffer
  ZG_ETH_HDR* eth_hdr = &eth_hdr_data; //just to remain consistent in notations
  memset((void*)eth_hdr, 0, sizeof(ZG_ETH_HDR));
  
  DBG("\r\nIncoming packet\r\n");
  hexdump(buf, len);
  
  ZG_RX_HDR* rx_hdr = (ZG_RX_HDR*)buf;
  
  memcpy( eth_hdr->dest, rx_hdr->dest, ZG_MACADDR_LEN );
  memcpy( eth_hdr->src, rx_hdr->src, ZG_MACADDR_LEN );
  eth_hdr->type = rx_hdr->type; //Lwip will convert the type in LE
  
  DBG("\r\nZg header\r\n");
  hexdump((byte*)rx_hdr, sizeof(ZG_RX_HDR));
  
  DBG("\r\nEth header\r\n");
  hexdump((byte*)eth_hdr, sizeof(ZG_ETH_HDR));
  
  //Since the ZG_ETH_HDR header is smaller than the ZG_RX_HDR one, we can copy the new header before the payload to get a full ethernet packet
  eth_buf = buf + sizeof(ZG_RX_HDR) - sizeof(ZG_ETH_HDR);
  eth_len = len - sizeof(ZG_RX_HDR) + sizeof(ZG_ETH_HDR);
  memcpy( eth_buf, (void*)eth_hdr, sizeof(ZG_ETH_HDR) );
  
  //Filter on packet type
  
  switch (htons(eth_hdr->type)) {
  //IP or ARP packet
  case ETHTYPE_IP:
  case ETHTYPE_ARP:
    break; //OK
  default:
    return; //We do not know how to handle this
  }
  
  #if ETH_PAD_SIZE
  eth_len += ETH_PAD_SIZE; /* allow room for Ethernet padding */
  #endif
  
  //Now we can pass this nice ethernet packet to lwip 
  //Allocate a buffer
  frame = pbuf_alloc(PBUF_RAW, eth_len, PBUF_POOL);
  if(frame == NULL)
  {
    //Out of memory
    return;
  }
  p = frame;
  
  #if ETH_PAD_SIZE
    pbuf_header(frame, -ETH_PAD_SIZE); /* drop the padding word */
  #endif
  
  //Copy buffer into lwip memory pool
  while( eth_len > 0 )
  {
    memcpy(p->payload, eth_buf, p->len);
    eth_buf += p->len;
    eth_len -= p->len;
    p = p->next;
    if( p == NULL )
    {
      //Should not happen
      break;
    }
  }

  if ( ethernet_input(frame, zg_netif) != ERR_OK ) //Pass the frame to lwip
  { 
    LWIP_DEBUGF(NETIF_DEBUG, ("ethernetif_input: IP input error\n"));
    pbuf_free(frame);
    frame = NULL;
  }
}


err_t zg_net_init(struct netif *netif) {
  LWIP_ASSERT("netif != NULL", (netif != NULL));
  
  memset(netif, 0, sizeof(struct netif));
  
  NETIF_INIT_SNMP(netif, snmp_ifType_ethernet_csmacd, 1500/*ZG_FIFO_BUF_SIZE*10*/);
  
  /* maximum transfer unit */
  netif->mtu = 1500;//ZG_FIFO_BUF_SIZE;
  
  /* set MAC hardware address length */
  netif->hwaddr_len = ETHARP_HWADDR_LEN;

  /* set MAC hardware address */
  if( !zg_data_mask.mac_addr ) //Get MAC addr from chip if not known yet
  {
    zg_mgmt_get_param(ZG_FIFO_MGMT_PARM_MACAD);
  }
  
  memcpy(netif->hwaddr, zg_data.mac_addr, ETHARP_HWADDR_LEN);
  
  /* device capabilities */
  /* don't set NETIF_FLAG_ETHARP if this device is not an ethernet one */
  netif->flags = NETIF_FLAG_BROADCAST | NETIF_FLAG_ETHARP | NETIF_FLAG_LINK_UP | NETIF_FLAG_IGMP;

  netif->state = NULL;

  netif->name[0] = IFNAME0;
  netif->name[1] = IFNAME1;

  /* We directly use etharp_output() here to save a function call.
   * You can instead declare your own function an call etharp_output()
   * from it if you have to do some checks before sending (e.g. if link
   * is available...) */
  netif->output          = etharp_output;
  netif->linkoutput      = zg_output;
  
  zg_netif = netif;

  return ERR_OK;
}

#endif
