
/*
Copyright (c) 2010 Donatien Garnier (donatiengar [at] gmail [dot] com)
 
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:
 
The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.
 
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "RWDModule.h"

RWDModule::RWDModule(PinName tx, PinName rx, PinName cts) : m_serial(tx, rx), m_cts(cts),
m_cmd(0), m_paramsBuf(NULL), m_respBuf(NULL), m_pos(0), m_paramsLen(0), m_respLen(0), m_ackOk(0), m_ackOkMask(0), m_ack(0), m_state(READY)
{
  //Setup baudrate
  m_serial.baud(9600);
  //Setup interrupts
  m_serial.attach(this, &RWDModule::intTx, Serial::TxIrq); //Serial port writeable
  m_serial.attach(this, &RWDModule::intRx, Serial::RxIrq); //Serial port readable
  m_cts.fall(this, &RWDModule::intClearToSend); //Clear To Send: can send a command
}

RWDModule::~RWDModule()
{

}

void RWDModule::command(uint8_t cmd, const uint8_t* params, int paramsLen, uint8_t* resp, size_t respLen, uint8_t ackOk, size_t ackOkMask) //Ack Byte is not included in the resp buf
{
  if(!ready()) //If reader is not ready, does not submit another command yet
    return;
  
  //Setup command
  m_cmd = cmd;
  
  //Setup parameters
  m_paramsBuf = (uint8_t*) params;
  m_paramsLen = paramsLen;
  
  //Setup response
  m_respBuf = resp;
  m_respLen = respLen;
  
  //Pos in buf is 0
  m_pos = 0;
  
  //Setup ack requirements
  m_ackOk = ackOk;
  m_ackOkMask = ackOkMask;
  
  m_state = CMD_QUEUED;
}
  
bool RWDModule::ready()
{
  return (m_state==READY);
}
   
bool RWDModule::result(uint8_t* pAck /*= NULL*/)
{
  if(!ready()) //Has command returned yet?
    return false;
  if(pAck) //If pointer is passed, return reader's ack
    *pAck = m_ack;
  return ((m_ack & m_ackOkMask) == m_ackOk); //Return whether the reader returned an error or OK ack
}

void RWDModule::intClearToSend()
{
  //Start sending command when Clear To Send falls
  if(m_state == CMD_QUEUED) //Is there a command to be sent?
  {
    m_state = SENDING_CMD;
    intTx(); //Start sending command
  }
}


void RWDModule::intTx()
{
  if(m_state != SENDING_CMD)
    return;  
  if(m_pos==0) //Must send command-byte first
    m_serial.putc((char)m_cmd);
  while(true) //Send payload
  {
    if(m_pos >= m_paramsLen) //Payload sent completely?
    {
      m_pos = 0;
      m_state = WAITING_FOR_ACK; //Next step
      return;
    }
    m_serial.putc((char)m_paramsBuf[m_pos]); //Send payload byte
    m_pos++;
  }
}

void RWDModule::intRx()
{
  if(m_state == WAITING_FOR_ACK) //Get answer
  {
    m_ack = m_serial.getc(); //Get Ack
    if( (m_ack & m_ackOkMask) != m_ackOk ) //Check if an error is returned
    {
      m_state = READY; //If yes, transfer is completed and result() will return false
      return;
    }
    if(m_respLen)
    {
      m_state = RECEIVING_ACK; //Ack OK, now need to get response
    }
    else
    {
      m_state = READY; //Ack OK, end of transfer
      return;
    }
  }
  if(m_state != RECEIVING_ACK) //Error, should not happen
  {
    while(m_serial.readable())
      m_serial.getc(); //Dump these bytes
    return; 
  }
  while(m_serial.readable()) //Read payload
  {
    m_respBuf[m_pos] = (uint8_t) m_serial.getc(); //Read byte and put it in resp buf
    m_pos++;
    if(m_pos >= m_respLen)
    {
      m_pos = 0;
      m_state = READY; //End of transfer, response retrieved with success
    }
  }
}
