/*
Copyright (c) 2010 ARM Limited

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:
 
The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.
 
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

/*
This is the main file for this project.
*/

//Core libs
#include "mbed.h"

//Config
#include "Config_Common.h"
#include "Config_Impl.h"

//RFID
#include "RWDMifare.h"

//Network
#include "NTPClient.h" //To setup RTC
//The MySQL Client is included in logger.h

//Logging
#include "taginfo.h"
#include "logger.h"

//UI
#include "beep.h"

#define MAX_UID_LEN 10 //Mifare max UID length is 10 bytes (triple-uid)

//Implemented by the interface-specific demos
int network_init();
int network_close();

RWDMifare reader(p13, p14, p12);
Serial pc (USBTX,USBRX);

DigitalIn exitBtn(p19);

Beep beep(p20);

NTPClient ntp;
Logger logger(SQL_TABLE);

extern "C" void HardFault_Handler() { printf("Hard Fault!\n"); error(""); }

//Main routine
int main() {
  pc.baud(115200); //Set bitrate for debug output
  
  //Init Mifare Reader
  RWDMifare::RWDMifareErr readerErr = reader.init();
  if(readerErr)
  {
    beep.beep(Beep::TONE_ERR);
    printf("Could not init reader (error %d)\n", readerErr);
    return -1;
  }
  
  //Open network connection
  int netRc = network_init();
  if(netRc)
  {
    beep.beep(Beep::TONE_ERR);
    printf("Could not init network interface\n");
    return -1;
  }

  //Setup the mbed's RTC
  NTPResult ntpResult;
  Host ntpServer(IpAddr(), 123, NTP_SERVER);
  do
  {
    ntpResult = ntp.setTime(ntpServer);
    if(ntpResult)
    {
      beep.beep(Beep::TONE_WARN);
      printf("Could not get time...\n");
      wait(5.);  
    }
  } while(ntpResult);
 
  //Check time
  time_t ctTime = time(NULL);  
  printf("Time is now (UTC) : %s\n", ctime(&ctTime)); 
   
  Host sqlServer(SQL_SERVER_IP, 3306, SQL_SERVER);
  
  const int locationId = LOCATION_ID; //This is the location id that will be sent on each req to the SQL server
 
  printf("Starting main loop...\n");

  bool connected = false;
  
  #if SDCARD
  logger.fileOpen(LOG_FILE); //Open file on SD card
  #endif
  
  beep.beep(Beep::TONE_OK);
  
  uint8_t uid[MAX_UID_LEN]; //Buffer for uid
  uint8_t lastUid[MAX_UID_LEN] = {0}; //Buffer for previous uid
  size_t uidLen;
  size_t lastUidLen = 0;
  
  while (!exitBtn.read()) //Main loop
  {
    if((!logger.isEmpty())&&(!logger.isConnecting())&&(!logger.isConnected())) //If logger is not ready, check its status
    {
      if(logger.connectionError()) //Error connecting to the server?
      {
        printf("Connection error (error code %d).\n", logger.getLastResult());
        beep.beep(Beep::TONE_WARN);
      }
      printf("Connecting to SQL Server...\n");
      logger.sqlOpen(sqlServer, SQL_USER, SQL_PASSWORD, SQL_DB); //Open a connection to server
      connected = false;
    }
    if((!connected)&&logger.isConnected()) //We are now connected
    {
      connected = true;
      printf("Connected to SQL Server...\n");
      beep.beep(Beep::TONE_OK);
    }
    if(!reader.getUID(uid, &uidLen)) //Got an UID successfully
    {
      uidLen = (uidLen<10)?uidLen:10; //Check length
      if( (uidLen != lastUidLen) || !!memcmp(lastUid, uid, uidLen) ) //Compare UID with previous one
      {
        beep.beep(Beep::TONE_INFO);
        printf("Tag ID = ");
        for(int i = 0; i < uidLen; i++)
          printf("%02x ", uid[i]);
        printf("\n");

        time_t timestamp = time(NULL); 
        timestamp += TIME_ZONE*3600;
        TagInfo tagInfo(uid, uidLen, locationId, timestamp); //Setup infos for data logging
        logger.log(&tagInfo); //Pass data to logger
        
        //Save uid to avoid duplicate db entries
        memcpy(lastUid, uid, uidLen);
        lastUidLen = uidLen;
      }
    }
    else //No card present
    {
      lastUidLen = 0; //Last logged card is now out of field
    }
    logger.service(); //Process logger service
  }
  
  printf("Stopping...\n");
  
  //Close everything nicely
  logger.sqlClose();
  
  #if SDCARD
  logger.fileClose();
  #endif
  
  network_close();
  
  printf("End.\n");
  beep.beep(Beep::TONE_OK);
  
  while(1);
  
}
