/*
Copyright (c) 2010 ARM Limited

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:
 
The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.
 
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

/*
This is a generic queue implementation to avoid using std::queue.
*/


#ifndef QUEUE_H
#define QUEUE_H

#include "string.h" //For size_t

namespace mbed {

template<class T>
class queue
{
public:
  queue(const size_t size) : m_size(size+1)
  {
    m_table = new T*[size + 1];
    m_backPos = m_frontPos = 0;
  }
  
  ~queue()
  {
    delete[] m_table;
  }
  
  void push(const T& item)
  {
    if(size() >= (m_size - 1))
      return;
    m_table[m_backPos] = new T(item);
    m_backPos++;
    if(m_backPos>=m_size)
      m_backPos = 0;
  }
  
  void pop()
  {
    if(empty())
      return;
    delete m_table[m_frontPos];
    m_frontPos++;
    if(m_frontPos>=m_size)
      m_frontPos = 0;
  }
  
  T& front()
  {
    return *m_table[m_frontPos];  
  }
  
  T& back()
  {
    return *m_table[m_backPos];
  }
  
  size_t size() const
  {
    if(m_backPos >= m_frontPos)
      return (m_backPos - m_frontPos);
    else
      return m_size - (m_frontPos - m_backPos);
  }
  
  bool empty() const
  {
    return !size();
  }
  
private:
  T** m_table;
  uint32_t m_frontPos; //Position of front element
  uint32_t m_backPos; //Position of last element
  
  size_t m_size;
  
};

}

#endif
