/*
Copyright (c) 2010 ARM Limited

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:
 
The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.
 
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "mbed.h"
#include "logger.h"

#include "MySQLClient.h"

#define MAX_ELEMENTS_IN_QUEUE 50


Logger::Logger(const string& table) : m_lSQLReq(MAX_ELEMENTS_IN_QUEUE), m_fd(NULL), m_sqlClient(), m_sqlLastResult(MYSQL_PROCESSING), 
m_sqlConnecting(false), m_sqlConnected(false), m_sqlConnectionError(false), m_sqlCommand(false)
{
  m_table = table;
}

Logger::~Logger()
{

}

//Open file for logging
int Logger::fileOpen(const char* filename)
{
  if(m_fd)
    return -1; //File already opened
  m_fd = fopen(filename, "a");
  if(!m_fd) //Try again in "w" mode (creates the file if not existent)
    m_fd = fopen(filename, "w");
  if(!m_fd)
    return -1; //Could not open file
  return 0;
}

//Close log file
int Logger::fileClose()
{
  if(!m_fd)
    return -1; //Nothing to close
  fclose(m_fd);
  m_fd = NULL;
  return 0;
}

//Try to connect to server
int Logger::sqlOpen(Host& host, const string& user, const string& password, const string& db)
{
  m_sqlClient.open(host, user, password, db, this, &Logger::onMySQLResult); //Non blocking
  //The onMySQLResult() method will be called on result of this command
  m_sqlConnecting = true;
  m_sqlConnectionError = false;
  return 0;
}
  
//Close server connection
int Logger::sqlClose()
{
  m_sqlClient.exit();
  m_sqlConnected = false;
  return 0;
}
  
//Log some data
int Logger::log(LogInfo* pInfo) //Returns 0 on success, -1 on failure
{

  if(m_fd) //If file is open, append record to file in CSV format
  {
    fputs(pInfo->toCSV().c_str(), m_fd);
    fputs("\r\n", m_fd); //Append new line
  }
  if(m_lSQLReq.size()<MAX_ELEMENTS_IN_QUEUE) //Check if there is (arbitrary) space in queue
  {
    m_lSQLReq.push(pInfo->toSQL(m_table)); //Put record in queue for next network push
    return 0; //Request queued
  }
  else
  {
    return -1; //Error
  }
}

void Logger::service()
{
  Net::poll(); //Poll Networking stack
  if(m_sqlConnected) //Are we connected to the server?
  {
    //Can serve SQL requests
    if(m_sqlCommand && m_sqlLastResult!=MYSQL_PROCESSING) //Previous command completed?
    {
      if(m_sqlLastResult==MYSQL_OK) //Has previous command completed with success?
      {
        if(!m_lSQLReq.empty())
        {
          m_lSQLReq.pop(); //Ok, we can dequeue previous element
        } 
      }
      m_sqlCommand = false; //No command being executed
    }
    if(!m_sqlCommand) //Ready for a command?
    {
      if(!m_lSQLReq.empty()) //Anything in the queue?
      {
        //Execute the next SQL request
        m_sqlLastResult = m_sqlClient.sql(m_lSQLReq.front()); //MYSQL_PROCESSING
        m_sqlCommand = true;
      }
    }
  }
}

bool Logger::isEmpty()
{
  return m_lSQLReq.empty();
}

bool Logger::isConnecting()
{
  return m_sqlConnecting;
}

bool Logger::isConnected()
{
  return m_sqlConnected;
}

bool Logger::connectionError()
{
  return m_sqlConnectionError;
}

MySQLResult Logger::getLastResult() //Returns last command's result
{
  return m_sqlLastResult;
}

void Logger::onMySQLResult(MySQLResult r) //Callback from MySQL client
{
  //Update status
  m_sqlLastResult = r;
  if(m_sqlConnecting) //Check if we're connecting, if so this result is the result of the connection
  {
    m_sqlConnected = (r==0)?true:false; //If no error, we're connected
    m_sqlConnecting = false;
    if(r!=0)
      m_sqlConnectionError = true; //Else there was an error during connection
  }
  if( (r==MYSQL_DNS) || (r==MYSQL_PRTCL) || (r==MYSQL_AUTHFAILED) || (r==MYSQL_TIMEOUT) || (r==MYSQL_CONN) ) //Was this a fatal error?
  {
    //Connection has been closed, we must reconnect
    m_sqlConnected = false;
    m_sqlConnectionError = true;
  }
}