/* Ethernetnetwork.cpp */
/*
 Copyright (C) 2012 ARM Limited.

 Permission is hereby granted, free of charge, to any person obtaining a copy of
 this software and associated documentation files (the "Software"), to deal in
 the Software without restriction, including without limitation the rights to
 use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
 of the Software, and to permit persons to whom the Software is furnished to do
 so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in all
 copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 SOFTWARE.
 */

#define __DEBUG__ 4
#ifndef __MODULE__
#define __MODULE__ "EthernetNetwork.cpp"
#endif

#include "core/fwk.h"

#include "EthernetNetwork.h"

#include "lwip/inet.h"
#include "lwip/netif.h"
#include "netif/etharp.h"
#include "lwip/dhcp.h"
#include "lwip/arch/lpc17_emac.h"
#include "lpc_phy.h" /* For the PHY monitor support */
#include "lwip/tcpip.h"

#include "mbed.h"

#if NET_ETHERNET

EthernetNetwork::EthernetNetwork() : LwIPInterface(), m_supervisor(&EthernetNetwork::phySupervisorCb, osTimerPeriodic, this), m_lpcNetif(), m_useDHCP(false)
{

}

int EthernetNetwork::init() //With DHCP
{
  ip_addr_t ip_n, mask_n, gateway_n;

  m_useDHCP = true;

  DBG("Initializing LwIP");
  LwIPInterface::init(); //Init LwIP, NOT including PPP

  DBG("DHCP IP assignment");

  memset((void*)&m_lpcNetif, 0, sizeof(m_lpcNetif));
  netif_add(&m_lpcNetif, NULL, NULL, NULL, NULL, lpc_enetif_init, ethernet_input/*tcpip_input???*/);
  netif_set_default(&m_lpcNetif);

  return OK;
}

int EthernetNetwork::init(const char* ip, const char* mask, const char* gateway,
    const char* dns1, const char* dns2) //No DHCP
{
  ip_addr_t ip_n, mask_n, gateway_n, dns1_n, dns2_n;

  m_useDHCP = false;

  DBG("Initializing LwIP");
  LwIPInterface::init(); //Init LwIP, NOT including PPP

  DBG("Static IP assignment");
  inet_aton(ip, &ip_n);
  inet_aton(mask, &mask_n);
  inet_aton(gateway, &gateway_n);
  inet_aton(dns1, &dns1_n);
  inet_aton(dns2, &dns2_n);

  memset((void*)&m_lpcNetif, 0, sizeof(m_lpcNetif));
  netif_add(&m_lpcNetif, &ip_n, &mask_n, &gateway_n, NULL, lpc_enetif_init, ethernet_input/*tcpip_input???*/);
  netif_set_default(&m_lpcNetif);

  return OK;
}

/*virtual*/int EthernetNetwork::connect()
{
  m_supervisor.start(250);
  netif_set_up(&m_lpcNetif);

  DBG("Enable MAC interrupts");
  NVIC_SetPriority(ENET_IRQn, ((0x01 << 3) | 0x01));
  NVIC_EnableIRQ(ENET_IRQn);

  if(m_useDHCP)
  {
    dhcp_start(&m_lpcNetif);
  }

  return OK;
}

/*virtual*/int EthernetNetwork::disconnect()
{
  if(m_useDHCP)
  {
    dhcp_stop(&m_lpcNetif);
  }

  netif_set_down(&m_lpcNetif);

  DBG("Disable MAC interrupts");
  NVIC_DisableIRQ(ENET_IRQn);

  m_supervisor.stop();

  return OK;
}

/*static*/ void EthernetNetwork::phySupervisorCb(void const* ctx)
{
  EthernetNetwork* pIf = (EthernetNetwork*) ctx;
  /* Call the PHY status update state machine once in a while
   to keep the link status up-to-date */
  if (lpc_phy_sts_sm(&pIf->m_lpcNetif) != 0)
  {
    /* Set the state of the LED to on if the ethernet link is
     active or off is disconnected. */
    if (pIf->m_lpcNetif.flags & NETIF_FLAG_LINK_UP)
    {
      pIf->m_linkUp = true;
    }
    else
    {
      pIf->m_linkUp = false;
    }
  }
}

#endif

