/* cpp_sockets_test.cpp */
/*
Copyright (C) 2012 ARM Limited.

Permission is hereby granted, free of charge, to any person obtaining a copy of
this software and associated documentation files (the "Software"), to deal in
the Software without restriction, including without limitation the rights to
use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
of the Software, and to permit persons to whom the Software is furnished to do
so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
*/

#define __DEBUG__ 4 //Maximum verbosity
#ifndef __MODULE__
#define __MODULE__ "cpp_sockets_test.cpp"
#endif

#include "core/fwk.h"
#include "mbed.h"

#include "rtos.h"

#include "EthernetInterface.h"
#include "TCPSocket.h"
#include "UDPSocket.h"

#define CHECK_ERR(ret) if(ret < 0) break;

DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);
DigitalOut led4(LED4);
void notify(bool a, bool b, bool c)
{
  led1 = a;
  led2 = b;
  led3 = c;
}

extern "C" void HardFault_Handler()
{
  error("Hard Fault!\n");
}

void test(void const*)
{
  EthernetInterface eth;
  DBG("Hello!");
  
  int count = 0;
  eth.init(); //Use DHCP

  count++;
  DBG("iteration #%d", count);
  
  char in_buf[256];

  notify(0, 1, 1);
  int ret = eth.connect();
  notify(0, 1, 0);
  if (ret == OK)
  {
    //TCP Socket test
    do
    {
      DBG("TCP Client test");
      TCPSocket sock;
      //http://mbed.org/media/uploads/donatien/hello.txt
      ret = sock.connect("mbed.org", 80);
      CHECK_ERR(ret);
      const char http_cmd[] = "GET /media/uploads/donatien/hello.txt HTTP/1.1\r\nHost: %s\r\n\r\n";
      ret = sock.send((std::uint8_t*)http_cmd, sizeof(http_cmd) - 1, 3000);
      CHECK_ERR(ret);
      
      bool firstIt = true;
      do
      {
        ret = sock.receive((std::uint8_t*)in_buf, 255, firstIt?3000:0);
        CHECK_ERR(ret);
        in_buf[ret] = '\0';
        DBG("Received %d chars from server: %s", ret, in_buf);
        firstIt = false;
      } while( ret > 0 );
      CHECK_ERR(ret);
      ret = sock.close();  
      CHECK_ERR(ret);
    } while(0);
    if(ret < 0)
    {
      WARN("TCP Client test failed");
    }
    else
    {
      DBG("TCP Client test succeeded");
    }
    
    do
    {
      DBG("TCP Server test");
      TCPSocket sock;
      ret = sock.bind(80); //Listen on all interfaces
      CHECK_ERR(ret);
      
      TCPSocket hdlrSock;
      ret = sock.listen(1);
      CHECK_ERR(ret);
      DBG("Now listening on port 80, open a browser and try to fetch the page");
      
      char* inHost;
      int inPort;
      ret = sock.accept(hdlrSock, &inHost, &inPort, 150000);
      CHECK_ERR(ret);
      DBG("Connection from %s on port %d", inHost, inPort);
      bool firstIt = true;
      do
      {
        ret = hdlrSock.receive((std::uint8_t*)in_buf, 255, firstIt?3000:0);
        CHECK_ERR(ret);
        in_buf[ret] = '\0';
        DBG("Received %d chars from client: %s", ret, in_buf);
        firstIt = false;
      } while( ret > 0 );
      CHECK_ERR(ret);
      
      const char http_resp[] = "HTTP/1.1 200 OK\r\nContent-Type: text/plain\r\nContent-Length: 12\r\n\r\nHello world!";
      ret = hdlrSock.send((std::uint8_t*)http_resp, sizeof(http_resp) - 1, 3000);
      CHECK_ERR(ret); 
      
      ret = hdlrSock.close();
      CHECK_ERR(ret);
      
      ret = sock.close();  
      CHECK_ERR(ret);
    } while(0);
    if(ret < 0)
    {
      WARN("TCP Server test failed");
    }
    else
    {
      DBG("TCP Server test succeeded");
    }
    
    //UDP Socket test
    do
    {
      DBG("UDP Client test");
      UDPSocket sock;
      sock.bind(0); //Use a random port
      const char daytime_cmd[] = "plop"; //Does not matter
      ret = sock.sendTo((std::uint8_t*)daytime_cmd, sizeof(daytime_cmd) - 1, "utcnist.colorado.edu", 37, 3000);
      CHECK_ERR(ret);

      char* inHost;
      int inPort;
      ret = sock.receiveFrom((std::uint8_t*)in_buf, 4, &inHost, &inPort, 3000);
      CHECK_ERR(ret);
      
      std::uint32_t timeRes = ntohl( *((std::uint32_t*)in_buf));
      
      DBG("Received %d bytes from server %s on port %d: %u seconds since 1/01/1900 00:00 GMT", ret, inHost, inPort, timeRes);
        
      ret = sock.close();  
      CHECK_ERR(ret);
    } while(0);
    if(ret < 0)
    {
      WARN("UDP Client test failed");
    }
    else
    {
      DBG("UDP Client test succeeded");
    }
    
  }
  eth.disconnect();
  DBG("Disconnected");

  notify(1, 1, 1);

  while (1)
  {
    Thread::wait(100);
  }
}

void keepAlive(void const*)
{
  while (1)
  {
    led1 = !led1;
    Thread::wait(500);
  }
}

void tick()
{
  led4 = !led4;
}

int main()
{
  Ticker t;
  t.attach(tick, 1);
  DBG_INIT();
  notify(1, 0, 0);

  Thread testTask(test, NULL, osPriorityNormal, 1024 * 4);
  keepAlive(NULL);

  return 0;
}
