#ifndef Controller_H
#define Controller_H

#include <bitset>
#include "mbed.h"



// Forward declaration of the classes that we use from the mbed library
// This avoids the need for us to include the huge mbed.h header inside our
// own library API
namespace mbed
{
class AnalogIn;
class InterruptIn;
class PwmOut;
class Timeout;
}


/** Controller Class
@brief This Class allows the joystick, buttons and petiometers to be implimented easily and coherently in other classes
@author Dr. Craig Evans, Modified by Dominic Kay
@date April 2017
*/
class Controller
{
    public:
    /** Controller events 
 * @brief List of all events which are triggered by an interrupt.
 */
enum ControllerEvent {
    A_PRESSED,     ///< Button A has been pressed
    B_PRESSED,     ///< Button B has been pressed
    X_PRESSED,     ///< Button X has been pressed
    Y_PRESSED,     ///< Button Y has been pressed
    L_PRESSED,     ///< Button L has been pressed
    R_PRESSED,     ///< Button R has been pressed
    BACK_PRESSED,  ///< Button "Back" has been pressed
    START_PRESSED, ///< Button "Start" has been pressed
    JOY_PRESSED,   ///< Joystick button has been pressed
    N_EVENTS       ///< A dummy flag that marks the end of the list 
};
private:
    mbed::PwmOut *_led1;
    mbed::PwmOut *_led2;
    mbed::PwmOut *_led3;
    mbed::PwmOut *_led4;
    mbed::PwmOut *_led5;
    mbed::PwmOut *_led6;

    mbed::InterruptIn *_button_A;
    mbed::InterruptIn *_button_B;
    mbed::InterruptIn *_button_X;
    mbed::InterruptIn *_button_Y;
    mbed::InterruptIn *_button_L;
    mbed::InterruptIn *_button_R;
    mbed::InterruptIn *_button_back;
    mbed::InterruptIn *_button_start;
    mbed::InterruptIn *_button_joystick;

    mbed::AnalogIn *_vert;
    mbed::AnalogIn *_horiz;

    mbed::PwmOut   *_buzzer;
    mbed::AnalogIn *_pot;

    mbed::Timeout *_timeout;

    std::bitset<N_EVENTS> _event_state; ///< A binary list of buttons that has been pressed
    float _x0;
    
public:

    /** Constructor */
    Controller();

    /** Destructor */
    ~Controller();

    /** Clear all the interrupt flags and turn off all LEDs */
    void init();

    /** Turn on all the LEDs */
    void ledsON();

    /** Turn off all the LEDs */
    void ledsOFF();

    /** Set all LEDs to duty-cycle EDIT
    *@param float value varing from 0.0 to 1.0
    */
    void leds(float val) const;

    /** Set LED to duty-cycle EDIT
    *@param led number (0 to 5)
    *@param value in range 0.0 to 1.0
    */
    void led(int n,float val) const;

    /** Read potentiometer
    *@returns value from 0.0 to 1.0 to effect other classes
    */
    float pot_value() const;

    /** Play tone on piezo
    * @param frequency of sound(Hz)
    * @param duration of sound(s)
    */
    void sound(float frequency, float duration);

    /**
     * @brief Checks if a specific flag has been set.
     * @param id[in] The specific flag ID
     * @return true if the flag occurred
     */
    bool check_event(ControllerEvent const id);

    /**
     * @brief Checks if joystick is flicked
     * @return x value
     */
     
    float get_joy();

    /**
     * @brief Allows for a specific integer to be returned if a specific flag has been set
     * @return interger unique to the flag set.
     */
    int check_for_buttons();



private:
    void init_buttons();
    void sound_off();

    void a_isr();
    void b_isr();
    void x_isr();
    void y_isr();
    void l_isr();
    void r_isr();
    void back_isr();
    void start_isr();
    void joy_isr();
};

#endif