#include "Controller.h"
#include "mbed.h"

//////////// constructor/destructor ////////////
Controller::Controller()
    :
    _led1(new PwmOut(PTA1)),
    _led2(new PwmOut(PTA2)),
    _led3(new PwmOut(PTC2)),
    _led4(new PwmOut(PTC3)),
    _led5(new PwmOut(PTC4)),
    _led6(new PwmOut(PTD3)),

    _button_A(new InterruptIn(PTB9)),
    _button_B(new InterruptIn(PTD0)),
    _button_X(new InterruptIn(PTC17)),
    _button_Y(new InterruptIn(PTC12)),
    _button_L(new InterruptIn(PTB18)),
    _button_R(new InterruptIn(PTB3)),
    _button_back(new InterruptIn(PTB19)),
    _button_start(new InterruptIn(PTC5)),
    _button_joystick(new InterruptIn(PTC16)),

    _vert(new AnalogIn(PTB10)),
    _horiz(new AnalogIn(PTB11)),

    _buzzer(new PwmOut(PTC10)),
    _pot(new AnalogIn(PTB2)),

    _timeout(new Timeout()),

    _event_state(0),
    _x0(0)
    
{}

Controller::~Controller()
{
    delete _led1,_led2,_led3,_led4,_led5,_led6;
    delete _button_A,_button_B,_button_joystick,_vert,_horiz;
    delete _button_X,_button_Y,_button_back,_button_start;
    delete _button_L,_button_R, _buzzer, _pot, _timeout;
}

///////////////// public methods /////////////////

void Controller::init()
{
    ledsOFF();
    init_buttons(); 

    // clear all flags
    _event_state = 0;  
    // read  joystick
    _x0 = _horiz->read();
}

void Controller::ledsOFF()
{
    leds(0.0);
}

void Controller::ledsON()
{
    leds(1.0);
}

void Controller::leds(float val) const
{
    if (val < 0.0f) {
        val = 0.0f;
    }
    if (val > 1.0f) {
        val = 1.0f;
    }
    
    // leds are active-low, so subtract from 1.0
    // 0.0 corresponds to fully-off, 1.0 to fully-on
    val = 1.0f - val;

    _led1->write(val);
    _led2->write(val);
    _led3->write(val);
    _led4->write(val);
    _led5->write(val);
    _led6->write(val);
}

void Controller::led(int n,float val) const
{
    // ensure they are within vlaid range
    if (val < 0.0f) {
        val = 0.0f;
    }
    if (val > 1.0f) {
        val = 1.0f;
    }
    
    switch (n) {
        
        // check for valid LED number and set value

        case 1:
            _led1->write(1.0f-val);   // active-low so subtract from 1
            break;
        case 2:
            _led2->write(1.0f-val);   // active-low so subtract from 1
            break;
        case 3:
            _led3->write(1.0f-val);   // active-low so subtract from 1
            break;
        case 4:
            _led4->write(1.0f-val);   // active-low so subtract from 1
            break;
        case 5:
            _led5->write(1.0f-val);   // active-low so subtract from 1
            break;
        case 6:
            _led6->write(1.0f-val);   // active-low so subtract from 1
            break;

    }
}

float Controller::pot_value() const
{
    return _pot->read();
}

void Controller::sound(float frequency, float duration)
{
    _buzzer->period(1.0f/frequency);
    _buzzer->write(0.5);  // 50% duty cycle - square wave
    _timeout->attach(callback(this, &Controller::sound_off), duration );
}

bool Controller::check_event(ControllerEvent const id)
{
    // Check whether event flag is set
    if (_event_state[id]) {
        _event_state.reset(id);  // clear flag
        return true;
    } else {
        return false;
    }
}

///////////////////// private methods ////////////////////////

void Controller::sound_off()
{
    // called after timeout
    _buzzer->write(0.0);
}

void Controller::init_buttons()
{
    // turn on pull-downs as other side of button is connected to 3V3
    // button is 0 when not pressed and 1 when pressed
    _button_A->mode(PullDown);
    _button_B->mode(PullDown);
    _button_X->mode(PullDown);
    _button_Y->mode(PullDown);
    _button_back->mode(PullDown);
    _button_start->mode(PullDown);
    _button_L->mode(PullDown);
    _button_R->mode(PullDown);
    _button_joystick->mode(PullDown);
    // therefore setup rising edge interrupts
    _button_A->rise(callback(this,&Controller::a_isr));
    _button_B->rise(callback(this,&Controller::b_isr));
    _button_X->rise(callback(this,&Controller::x_isr));
    _button_Y->rise(callback(this,&Controller::y_isr));
    _button_L->rise(callback(this,&Controller::l_isr));
    _button_R->rise(callback(this,&Controller::r_isr));
    _button_start->rise(callback(this,&Controller::start_isr));
    _button_back->rise(callback(this,&Controller::back_isr));
    _button_joystick->rise(callback(this,&Controller::joy_isr));
}

// button interrupts ISRs
// Each of these simply sets the appropriate event bit in the _event_state
// variable
void Controller::a_isr()
{
    _event_state.set(A_PRESSED);
}
void Controller::b_isr()
{
    _event_state.set(B_PRESSED);
}
void Controller::x_isr()
{
    _event_state.set(X_PRESSED);
}
void Controller::y_isr()
{
    _event_state.set(Y_PRESSED);
}
void Controller::l_isr()
{
    _event_state.set(L_PRESSED);
}
void Controller::r_isr()
{
    _event_state.set(R_PRESSED);
}
void Controller::back_isr()
{
    _event_state.set(BACK_PRESSED);
}
void Controller::start_isr()
{
    _event_state.set(START_PRESSED);
}
void Controller::joy_isr()
{
    _event_state.set(JOY_PRESSED);
}



// get raw joystick coordinate for x in range -1 to 1
// North     (0,1)
// South     (0,-1)

float Controller::get_joy()
{
    // read() returns value in range 0.0 to 1.0 so is scaled and centre value
    // substracted to get values in the range -1.0 to 1.0
    float x = 2.0f*( _horiz->read() - _x0 );
    return x;
}


int Controller::check_for_buttons() { //return specific interger depending on the interrupt triggered
    if (check_event(A_PRESSED)) { 
        return 1;  }
    else if (check_event(B_PRESSED)) { 
        return 2; }
    else if (check_event(X_PRESSED)) { 
        return 3; }
    else if (check_event(Y_PRESSED)) { 
        return 4; }
    else if (check_event(L_PRESSED)) { 
        return 5; } 
    else if (check_event(R_PRESSED)) { 
        return 6; }  
    else if (check_event(JOY_PRESSED)) { 
        return 7; }       
        else {     
        return 0; }  
    } 

