/*
 *  this is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  this software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with libfftpack; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
 
#include "mbed.h"
#include "USBHost6ChDac.h"
#include "USBHostMSD.h"
#include "filter.h"
#include "USBAudio.h"

#if defined(TARGET_RZ_A1H)
#include "usb_host_setting.h"
#else
#define USB_HOST_CH     0
#endif

//data size of receiving buffer in long word (32767samples 16bit stereo)
#define IN_DWORDS               32768                       
#define IN_BANKS                2
//data size of sending buffer in long word (32767samples 16bit 6ch)
#define OUT_DWORDS              (3 * IN_DWORDS)             
#define OUT_BANKS               2
//data size of silence data in long word (48ksps 16bit 6ch 100msec)
#define NULL_DWORDS             (48 * 2 * 6 * 100 / 4)      

//shows USB audio is attached or not
extern int USB_Audio_Attached;
//receiving buffer (2 banks)
static volatile uint32_t BufR[IN_BANKS][IN_DWORDS];
//sending buffer (2 banks)
static volatile uint32_t BufW[OUT_BANKS][OUT_DWORDS];
//silence data buffer to adjust buffer volume
static volatile uint32_t BufNull[NULL_DWORDS] = {};
//overflow flag of FFT
int FFTOverFlow = 0;                                        

//push button on the board
DigitalIn  button(USER_BUTTON0);                            

//shows adjustment for buffer short or full is done
DigitalOut LedR(LED1);                                      
DigitalOut LedG(LED2);
//shows FFT is in process
DigitalOut LedB(LED3);                                      

//USBDAC host
USBHostDac *pusbdac;

//receive buffer bank 0 or 1 ready
Semaphore semR0(0);
Semaphore semR1(0);
//send buffer bank 0 or 1 ready
Semaphore semW0(0);
Semaphore semW1(0);

Semaphore *semR[2] = {&semR0, &semR1}; 
Semaphore *semW[2] = {&semW0, &semW1}; 

//measures latency of semaphore waiting
Timer Tmr;

//target latency for buffer slack in mili sec (88msec for buffer full at 11 ITDs)
#define TARGET_LATENCY          40
//allowance of latency
//latency is controled in TARGET_LATENCY - LATENCY_ALLOWANCE to TARGET_LATENCY + LATENCY_ALLOWANCE
#define LATENCY_ALLOWANCE       30
//output byte number to get 1msecond adjustment (48ksps 16bit 6ch at 1milisec)
#define OUT_BYTE_PER_MILISEC    (48*2*6)    

//sound data sending task
void UsbSend(void const* arg) 
{
    //usb host
    USBHostDac * p_usbdac = (USBHostDac *)arg;
    //bank number of buffer (0 or 1)
    int Bank = 0;
    //waiting latency of semaphore
    int Latency;
    //adjust time in mili second
    int Adjmsec;
    //number of skipping byte 
    int SkipBytes;

    Tmr.start();
    for(;;) 
    {
        //riset adjusting display
        LedR = 0;
        //reset timer to measure latency of semaphore
        Tmr.reset();
        semW[Bank]->wait();
        //get latency
        Latency = Tmr.read_ms();
        printf("Latency:%d\n",Latency);
        //if semahore's latency is too large, data shortage has happen
        if(Latency > (TARGET_LATENCY + LATENCY_ALLOWANCE)) 
        {
            //get period for adjustment 
            Adjmsec = Latency - TARGET_LATENCY;
            if(Adjmsec > TARGET_LATENCY) Adjmsec = TARGET_LATENCY;
            printf("Recover %dmsec\n",Adjmsec);
            //put silence data into sending queue to prevent data shortage
            p_usbdac->send((uint8_t *)BufNull, Adjmsec * OUT_BYTE_PER_MILISEC, false);
            //data skip is unnecessary for buffer have some vacancy
            SkipBytes = 0;
            //Show adjusting happend
            LedR = 1;
        }
        //if semahore's latency is too small, buffer is nearly full
        else if(Latency < (TARGET_LATENCY - LATENCY_ALLOWANCE))
        {
            //get period for adjustment 
            Adjmsec = TARGET_LATENCY - Latency;
            if(Adjmsec > TARGET_LATENCY) Adjmsec = TARGET_LATENCY;
            //decrease sending data, to make some vacancy on the buffer
            SkipBytes = Adjmsec * OUT_BYTE_PER_MILISEC;
            printf("Skip %dmsec %dByte\n",Adjmsec,SkipBytes);
            //Show adjusting happend
            LedR = 1;
        }
        else
        {
            //no demand for data skip because buffer have regular vacancy
            SkipBytes = 0;
        }
        //send music data
        p_usbdac->send((uint8_t *)&BufW[Bank][SkipBytes / 4], OUT_DWORDS * 4 - SkipBytes, false);
        //next bank
        Bank++;
        Bank &= 1;
    }
}

//FFT solving task
void FFTSolve(void const* arg) 
{
    int Bank = 0;
    for(;;)  
    {  
        //wait for FFT sorce data ready
        semR[Bank]->wait();
        //show FFT is in work
        LedB = 1;
        //apply FIR filters and write the result on the buffer
        FFTOverFlow = FLT_Filter((int16_t *)(BufW[Bank]),(int16_t *)(BufR[Bank]),!button);
//      FFTOverFlow = FLT_Through((int16_t *)(BufW[Bank]),(int16_t *)BufR[Bank]);
        //show FFT ended
        LedB = 0;
        //order to send the FFT result 
        semW[Bank]->release();
        //next bank
        Bank++;
        Bank &= 1;
    }
}

//receive 32768 sample of audio input from USB audio device
int usbdevice_receive32k(USBAudio *paudio,uint32_t *Buf)
{
    int i;
    
    //repat reading in 1 second
    for(i = 0;i < 70;i++)
    {
        //attempting to read
        if(paudio->read32k(Buf))
        {
            //returns 1 if correctly read
            return 1;
        }
        else
        {
            //if read error, wait a moment and retry
            Thread::wait(10);
        }
    }
    //returns 0 if read error
    return 0;
}

int main() 
{
    //receiving bank (0 or 1)
    int Bank = 0;
    //receiving result
    int rc = 0;
    //input mode 1:USB audio device 0:LINE-IN of USB DAC
    int InModeUSB = 1;
    
    printf("\n\nStarted\n");

    //initializing silence buffer
    memset((uint8_t *)BufNull,0,NULL_DWORDS * 4);

    //USB audio device initialize
    static USBAudio audio(48000, 2, 8000, 1, 0x7180, 0x7500);

    //start Mass Strage Device host
    USBHostMSD msd("usb");
    //start Audio host
    USBHostDac usbdac;
    //get pointer to Audio host to use it in other function
    pusbdac = &usbdac;

    //wait connection of MSD
    while(!msd.connect()) 
    {
        printf("Attempting to connect a MSD\n");
        Thread::wait(500);
    }
    
    //initializing FIR filters (setting up FFT and reading filter coefficients)
    while(! FLT_Init())
    {
        printf("Filter initialization error\n");
        Thread::wait(500);
    }

    //create sending and FFT task
    Thread SendTask(UsbSend, &usbdac, osPriorityNormal, 1024 * 8);
    Thread FFTTask(FFTSolve, &usbdac, osPriorityNormal, 1024 * 8);

    for(;;)
    {
        //
        //      CONNECTION WITH USB-DAC
        //
        //if USB-DAC is disconnected
        if(! pusbdac->connected())
        {
            //connect USB-DAC
            pusbdac->connect(); 
        }
        //
        //      INPUT SWITCHING
        //
        //if now switched to USB_INPUT mode
        if(InModeUSB)
        {
            //if USB_INPUT is detached
            if(! USB_Audio_Attached)
            {
                //select LINE_INPUT for input
                printf("Switched to LINE-IN\n");
                InModeUSB = 0;
            }
        }
        //if now switched to LINE_INPUT mode
        else
        {
            //if USB_INPUT is attached
            if(USB_Audio_Attached)
            {
                //select USB_INPUT for input
                printf("Switched to USB-IN\n");
                InModeUSB = 1;
            }
        }
        //
        //      RECEIVING AUDIO DATA
        //
        //if using USB-INPUT
        if(InModeUSB)
        {
            //receive USB-IN
            rc = usbdevice_receive32k(&audio,(uint32_t *)BufR[Bank]);
        }
        //if using LINE-IN
        else
        {
            //receive LINE-IN
            rc = usbdac.receive((uint8_t *)(BufR[Bank]), IN_DWORDS * 4);
        }
        //
        //      ERROR HANDLING and ORDERING TO START FFT
        //
        //if error occurs
        if(! rc)
        {
            //order send task to full some data to prevent data short 
            //Recover = 1;
            //clear all buffers to mute
            audio.clear32k();
            FLT_Clear();
            memset((uint8_t *)BufR,0,IN_DWORDS * IN_BANKS * 4);
            memset((uint8_t *)BufW,0,OUT_DWORDS * OUT_BANKS * 4);
        }
        //if read correctly
        else
        {
            //order to start FFT
            semR[Bank]->release();
            //select next bank  
            Bank++;
            Bank &= 1;
        }
    }
}
