/* Copyright (c) 2010-2012 mbed.org, MIT License
*
* Permission is hereby granted, free of charge, to any person obtaining a copy of this software
* and associated documentation files (the "Software"), to deal in the Software without
* restriction, including without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in all copies or
* substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
* BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
* DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#define __DEBUG__ 0
#ifndef __MODULE__
#define __MODULE__ "WANDongle.cpp"
#endif

#include "core/dbg.h"
#include <cstdint>
#include "rtos.h"

#include "WANDongle.h"
#include "WANDongleInitializer.h"

WANDongle::WANDongle() : m_pInitializer(NULL), m_serialCount(0)
{
    m_lastDongle = NULL;
    host = USBHost::getHostInst();
    init();
    DBG("WANDongle object instantiated. getHostInst method called on USBHost.");
}


bool WANDongle::connected() {
  return dev_connected;
}

bool WANDongle::tryConnect()
{
  //FIXME should run on USB thread

  DBG("Trying to connect device");

  if (dev_connected) {
      return true;
  }
  
  m_pInitializer = NULL;

  host->lock();

  for (int i = 0; i < MAX_DEVICE_NB; i++)
  {
      if ((dev = host->getDevice(i)) != NULL)
      {
          m_pInitializer = NULL; //Will be set in setVidPid callback
      
          DBG("Found one device reset it");
          int ret = host->resetDevice(dev);
          if(ret)
          {
            host->unlock();
            return false;
          }

          DBG("Enumerate");
          ret = host->enumerate(dev, this);
          
          if(ret)
          {
            host->unlock();
            return false;
          }
          
          DBG("Device has VID:%04x PID:%04x", dev->getVid(), dev->getPid());
                   
          if(m_pInitializer) //If an initializer has been found
          {
            DBG("m_pInitializer=%p", m_pInitializer);
            DBG("m_pInitializer->getSerialVid()=%04x", m_pInitializer->getSerialVid());
            DBG("m_pInitializer->getSerialPid()=%04x", m_pInitializer->getSerialPid());
            if ((dev->getVid() == m_pInitializer->getSerialVid()) && (dev->getPid() == m_pInitializer->getSerialPid()))
            {
              DBG("The dongle is in virtual serial mode");
              host->registerDriver(dev, 0, this, &WANDongle::init);
              m_serialCount = m_pInitializer->getSerialPortCount();
              DBG("Num serial ports: %d",m_serialCount);
              if( m_serialCount > WANDONGLE_MAX_SERIAL_PORTS )
              {
                DBG("setting serial count %d to %d",m_serialCount,WANDONGLE_MAX_SERIAL_PORTS);
                m_serialCount = WANDONGLE_MAX_SERIAL_PORTS;
              }
              for(int j = 0; j < m_serialCount; j++)
              {
                DBG("Connecting serial port #%d", j+1);
                DBG("Ep %p", m_pInitializer->getEp(dev, j, false));
                DBG("Ep %p", m_pInitializer->getEp(dev, j, true));
                m_serial[j].connect( dev, m_pInitializer->getEp(dev, j, false), m_pInitializer->getEp(dev, j, true) );
              }
              
              DBG("Device connected");
              
              dev_connected = true;
              host->unlock();
              
              return true;
            }
            else if ((dev->getVid() == m_pInitializer->getMSDVid()) && (dev->getPid() == m_pInitializer->getMSDPid()))
            {
              DBG("Vodafone dongle detected in MSD mode");
              //Try to switch   
              if( m_pInitializer->switchMode(dev) )
              {
                DBG("Switched OK");
                host->unlock();
                return false; //Will be connected on a next iteration
              }
              else
              {
                ERR("Could not switch mode");
                host->unlock();
                return false;
              }
            }
          } //if()
      } //if()
      /*else
      {
        // Looks like the getDevice method failed and returned a null pointer. Maybe there has been a power cut, modem pulled or something to stop the modem.
        // Lets run the initilise routine again. Next time round the loop we might have a handle to an end point!
        DBG("Trying to initialise the USB stack since the getDevice method has returned a null pointer");
        init();
      }*/
  } //for()
  host->unlock();
  return false;
}

bool WANDongle::disconnect()
{
  dev_connected = false;
  for(int i = 0; i < WANDONGLE_MAX_SERIAL_PORTS; i++)
  {
    m_serial[i].disconnect();
  }
  return true;
}

WAN_DONGLE_TYPE WANDongle::getDongleType()
{
  if( m_pInitializer != NULL )
  {
    return m_pInitializer->getType();
  }
  else
  {
    return WAN_DONGLE_TYPE_UNKNOWN;
  }
}

IUSBHostSerial& WANDongle::getSerial(int index)
{
  return m_serial[index];
}

int WANDongle::getSerialCount()
{
  return m_serialCount;
}

//Private methods
void WANDongle::init()
{
  m_pInitializer = NULL;
  dev_connected = false;
  for(int i = 0; i < WANDONGLE_MAX_SERIAL_PORTS; i++)
  {
    m_serial[i].init(host);
  }
}


/*virtual*/ void WANDongle::setVidPid(uint16_t vid, uint16_t pid)
{
    //Load right initializer
  WANDongleInitializer** initializer = WANDongleInitializer::getInitializers(host);
  
  while(*initializer)
  {
    if (m_lastDongle)
    {
        DBG("Initializer has been already detected in previous step");
        m_pInitializer = m_lastDongle;    
        break;
    }
    
    
    DBG("*initializer=%p", *initializer);
    DBG("(*initializer)->getSerialVid()=%04x", (*initializer)->getSerialVid());
    DBG("(*initializer)->getSerialPid()=%04x", (*initializer)->getSerialPid());
    if ((dev->getVid() == (*initializer)->getSerialVid()) && (dev->getPid() == (*initializer)->getSerialPid()))
    {
      DBG("The dongle is in virtual serial mode");
      m_pInitializer = *initializer;
      break;
    }
    else if ((dev->getVid() == (*initializer)->getMSDVid()) && (dev->getPid() == (*initializer)->getMSDPid()))
    {
      DBG("Vodafone K3370 dongle detected in MSD mode");
      m_pInitializer = *initializer;
      
      //Store successfully detected modem type so that after switch to serial mode 
      //we can distinuguish modems with same Vid-Pid pairs
      m_lastDongle = *initializer;
      
      break;
    }
    initializer++;
  } //while()
  if(m_pInitializer)
  {
    m_pInitializer->setVidPid(vid, pid);
  }
}

/*virtual*/ bool WANDongle::parseInterface(uint8_t intf_nb, uint8_t intf_class, uint8_t intf_subclass, uint8_t intf_protocol) //Must return true if the interface should be parsed
{
  if(m_pInitializer)
  {
    return m_pInitializer->parseInterface(intf_nb, intf_class, intf_subclass, intf_protocol);
  }
  else
  {
    return false;
  }
}

/*virtual*/ bool WANDongle::useEndpoint(uint8_t intf_nb, ENDPOINT_TYPE type, ENDPOINT_DIRECTION dir) //Must return true if the endpoint will be used
{
  if(m_pInitializer)
  {
    return m_pInitializer->useEndpoint(intf_nb, type, dir);
  }
  else
  {
    return false;
  }
}
