#ifndef MAIN_H
#define MAIN_H
 
#include "mbed.h"
#include "rtos.h"
 
#define I1pin D2            //Photointerrupter input pins
#define I2pin D11
#define I3pin D12
 
#define CHApin   D7         //Incremental encoder input pins
#define CHBpin   D8
 
//Motor Drive output pins   //Mask in output byte
#define L1Lpin D4           //0x01
#define L1Hpin D5           //0x02
#define L2Lpin D3           //0x04
#define L2Hpin D6           //0x08
#define L3Lpin D9           //0x10
#define L3Hpin D10          //0x20
 
//for easier printing
#define PRINT(...) pc.printf(__VA_ARGS__); pc.printf("\n\r");
 
/*Mapping from sequential drive states to motor phase outputs
State   L1  L2  L3  Angle   [X,X,L3H,L3L,L2H,L2L,L1H,L1L]
0       H   -   L   0       0x12
1       -   H   L   60      0x18
2       L   H   -   120     0x09
3       L   -   H   180     0x21
4       -   L   H   240     0x24
5       H   L   -   300     0x06
6       -   -   -   X       0x00
7       -   -   -   X       0x00    */
//Drive state to output table. 0x00%6 = 0x06%6
volatile const int8_t driveTable[] = {0x12,0x18,0x09,0x21,0x24,0x06,0x00,0x00};
 
//Mapping from interrupter inputs to sequential rotor states.
/* sateMap[0], [7] are not valid, as ideally they never happen.
State   I3 I2 I1            Angle       //Not clear why this order.
0       1  1  1     0x07    X
1       1  0  1     0x05    0-60
2       0  1  1     0x03    240-300
3       1  0  0     0x04    60-120
4       0  0  1     0x01    300-0
5       1  1  0     0x06    120-180
6       0  1  0     0x02    180-240
7       1  1  1     0x07    X       */
volatile const int8_t stateMap[] = {0x07,0x05,0x03,0x04,0x01,0x00,0x02,0x07};
Mutex mutex;
//Mapping precision angle control states
const int8_t relativeStateMap[] = {0x00, 0x03, 0x01, 0x02};
// 2* 360/(117*4), the angular difference between adjacent states
const float angularResolution = 1.5385;
const float angle = 6.283;                  //2*pi for 1 revolution
 
//Phase lead to make motor spin. -2 equivalent to 4, 2 equivalent to 8
volatile int8_t lead = -2;                  //2 for forwards, -2 for backwards
 
volatile const int8_t orState = 3;          //measured hardware motor offset
 
volatile uint8_t state = 0;                 //Initial state
 
volatile int motorPWMPeriod = 40;                   //PWM motor period
volatile float velocityDuty = 0.0;
 
DigitalOut led1(LED1);                      //Status LED
 
InterruptIn I1(I1pin);                      //Photointerrupter inputs
InterruptIn I2(I2pin);
InterruptIn I3(I3pin);
 
InterruptIn CHA(CHApin);                    //Precision angle interrupt pins
InterruptIn CHB(CHBpin);
 
PwmOut L1L(L1Lpin);                         //Motor Drive outputs
PwmOut L1H(L1Hpin);
PwmOut L2L(L2Lpin);
PwmOut L2H(L2Hpin);
PwmOut L3L(L3Lpin);
PwmOut L3H(L3Hpin);
//      L1L, L1H, L2L, L2H, L3L, L3H

int8_t updateState() {           //Convert photointerrupter inputs to a 
    return stateMap[4*I3 + 2*I2 + I1];  // rotor state. Reports 1/6 positions
}
 
PwmOut *LPins[6];
float PinOff[6] = {0.0, 1.0, 0.0, 1.0, 0.0, 1.0};

#endif