/*******************************************************************************
* DISCLAIMER
* This software is supplied by Renesas Electronics Corporation and is only
* intended for use with Renesas products. No other uses are authorized. This
* software is owned by Renesas Electronics Corporation and is protected under
* all applicable laws, including copyright laws.
* THIS SOFTWARE IS PROVIDED "AS IS" AND RENESAS MAKES NO WARRANTIES REGARDING
* THIS SOFTWARE, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING BUT NOT
* LIMITED TO WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
* AND NON-INFRINGEMENT. ALL SUCH WARRANTIES ARE EXPRESSLY DISCLAIMED.
* TO THE MAXIMUM EXTENT PERMITTED NOT PROHIBITED BY LAW, NEITHER RENESAS
* ELECTRONICS CORPORATION NOR ANY OF ITS AFFILIATED COMPANIES SHALL BE LIABLE
* FOR ANY DIRECT, INDIRECT, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES FOR
* ANY REASON RELATED TO THIS SOFTWARE, EVEN IF RENESAS OR ITS AFFILIATES HAVE
* BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
* Renesas reserves the right, without notice, to make changes to this software
* and to discontinue the availability of this software. By using this software,
* you agree to the additional terms and conditions found by accessing the
* following link:
* http://www.renesas.com/disclaimer*
* Copyright (C) 2015 Renesas Electronics Corporation. All rights reserved.
*******************************************************************************/

#ifndef SCAN_FOLDER_H
#define SCAN_FOLDER_H

#include "r_typedefs.h"
#include "FATFileSystem.h"

/*--- Macro definition of folder scan ---*/
#define MAX_FOLDER_NUM      (99u)       /* Supported number of folders */
#define MAX_TRACK_NUM       (999u)      /* Supported number of files  */
#define MAX_FOLDER_DEPTH    (8u)        /* Supported folder levels */
#define MAX_NAME_LENGTH     (NAME_MAX)  /* Maximum length of track name and folder name */
#define MAX_PATH_LENGTH     (511)       /* Maximum length of the full path */

class ScanFolder {
public:

    /** Initializes the folder structure
     *
     */
    void init();

    /** Scans the folder structure
     *
     *  @param p_root_name The character string to identify root directory.
     *  @param p_extension_tbl extension tbl
     *
     *  @returns 
     *    Results of process. true is success. false is failure.
     */
    bool scan(char_t * p_root_name, const char_t ** pp_extension_tbl);

    /** Gets the total number of detected files
     *
     *  @returns 
     *    the total number of files
     */
    uint32_t getTotalFile();

    /** Gets the file name
     *
     *  @param track_id Track ID [0 - (total track - 1)]
     *
     *  @returns 
     *    Pointer to the file name.
     */
    const char_t * getFileName(const uint32_t track_id);

    /** Opens the file
     *
     *  @param track_id Track ID [0 - (total track - 1)]
     *
     *  @returns 
     *    Pointer to the track handle
     */
    FILE * open(const uint32_t track_id);

    /** Closes the file
     *
     *  @param fp Pointer to the track handle
     */
    void close(FILE * const fp);

private:
    /* Information of a folder / track. */
    typedef struct {
        char_t      name[MAX_NAME_LENGTH + 1];      /* Name of a folder / track. */
        uint32_t    parent_number;                  /* Number of the parent folder. */
    } item_t;

    /* Information of folder scan in USB memory */
    typedef struct {
        item_t      folder_list[MAX_FOLDER_NUM];        /* Folder list */
        item_t      track_list[MAX_TRACK_NUM];          /* File list */
        uint32_t    total_folder;                       /* Total number of folders */
        uint32_t    total_file;                         /* Total number of files */
        char_t      work_buf[MAX_PATH_LENGTH + 1];      /* Work */
                                                        /* (Including the null terminal character.) */
    } fid_scan_folder_t;

    fid_scan_folder_t scan_data;

    const char_t * get_full_path(const item_t * const p_item);
    bool open_dir(const item_t * const p_item, FATFS_DIR * const p_fdir);
    bool read_dir(FATFS_DIR * const p_fdir,  const char_t ** const p_name, bool * const p_flag_dir);
    bool regist_item(item_t * const p_item, const char_t * const p_name, const uint32_t parent);
    bool check_extension(const char_t * const p_name, const char_t ** pp_extension_tbl);
    bool check_folder_depth(const uint32_t folder_id);
};

#endif /* SCAN_FOLDER_H */
