// ----------------------------------------------------------------------------
// Copyright 2016-2018 ARM Ltd.
//
// SPDX-License-Identifier: Apache-2.0
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
// ----------------------------------------------------------------------------
#ifndef MBED_TEST_MODE

#include "mbed.h"
#include "simple-mbed-cloud-client.h"
#include "FATFileSystem.h"
#include "EasyAttach_CameraAndLCD.h"
#include "dcache-control.h"
#include "JPEG_Converter.h"

/**** User Selection *********/
#define VIDEO_PIXEL_HW         (320u)  /* QVGA */
#define VIDEO_PIXEL_VW         (240u)  /* QVGA */
#define JPEG_ENCODE_QUALITY    (75)    /* JPEG encode quality (min:1, max:75 (Considering the size of JpegBuffer, about 75 is the upper limit.)) */
/*****************************/

#define DATA_SIZE_PER_PIC      (2u)
#define FRAME_BUFFER_STRIDE    (((VIDEO_PIXEL_HW * DATA_SIZE_PER_PIC) + 31u) & ~31u)
#define FRAME_BUFFER_HEIGHT    (VIDEO_PIXEL_VW)

#if defined(__ICCARM__)
#pragma data_alignment=32
static uint8_t user_frame_buffer0[FRAME_BUFFER_STRIDE * FRAME_BUFFER_HEIGHT];
#pragma data_alignment=32
static uint8_t JpegBuffer[1024 * 32];
#else
static uint8_t user_frame_buffer0[FRAME_BUFFER_STRIDE * FRAME_BUFFER_HEIGHT]__attribute((aligned(32)));
static uint8_t JpegBuffer[1024 * 32]__attribute((aligned(32)));
#endif
static DisplayBase Display;

// An event queue is a very useful structure to debounce information between contexts (e.g. ISR and normal threads)
// This is great because things such as network operations are illegal in ISR, so updating a resource in a button's fall() function is not allowed
EventQueue eventQueue;

// Default block device
BlockDevice *bd = BlockDevice::get_default_instance();
FATFileSystem fs("fs");

// Default network interface object
NetworkInterface *net = NetworkInterface::get_default_instance();

InterruptIn btn(USER_BUTTON0);
// Declaring pointers for access to Pelion Device Management Client resources outside of main()
MbedCloudClientResource *button_res;
MbedCloudClientResource *pattern_res;

Thread jpegTask(osPriorityNormal, 1024 * 4);

void save_camera_image_req(void) {
    jpegTask.flags_set(1);
}

void jpeg_task(MbedCloudClientResource *resource) {
    M2MResource *m2m_camera_img_res = resource->get_m2m_resource();
    JPEG_Converter Jcu;
    JPEG_Converter::bitmap_buff_info_t buff_info;
    JPEG_Converter::encode_options_t   encode_opt;
    size_t encode_size;

    // Camera start
    EasyAttach_Init(Display);
    Display.Video_Write_Setting(
        DisplayBase::VIDEO_INPUT_CHANNEL_0,
        DisplayBase::COL_SYS_NTSC_358,
        (void *)user_frame_buffer0,
        FRAME_BUFFER_STRIDE,
        DisplayBase::VIDEO_FORMAT_YCBCR422,
        DisplayBase::WR_RD_WRSWA_32_16BIT,
        VIDEO_PIXEL_VW,
        VIDEO_PIXEL_HW
    );
    EasyAttach_CameraStart(Display, DisplayBase::VIDEO_INPUT_CHANNEL_0);

    // Jpeg setting
    buff_info.width              = VIDEO_PIXEL_HW;
    buff_info.height             = VIDEO_PIXEL_VW;
    buff_info.format             = JPEG_Converter::WR_RD_YCbCr422;
    buff_info.buffer_address     = (void *)user_frame_buffer0;
    encode_opt.encode_buff_size  = sizeof(JpegBuffer);
    encode_opt.input_swapsetting = JPEG_Converter::WR_RD_WRSWA_32_16_8BIT;
    Jcu.SetQuality(JPEG_ENCODE_QUALITY);

    while (true) {
        ThisThread::flags_wait_all(1);
        dcache_invalid(JpegBuffer, sizeof(JpegBuffer));
        if (Jcu.encode(&buff_info, JpegBuffer, &encode_size, &encode_opt) == JPEG_Converter::JPEG_CONV_OK) {
            m2m_camera_img_res->set_value(JpegBuffer, encode_size);
            printf("Camera image update: %dbyte\n", encode_size);
        } else {
            printf("Jpeg convert error\n");
        }
    }
}

void button_press() {
    int v = button_res->get_value_int() + 1;

    button_res->set_value(v);

    printf("User button clicked %d times\n", v);    

    save_camera_image_req();
}

/**
 * PUT handler
 * @param resource The resource that triggered the callback
 * @param newValue Updated value for the resource
 */
void pattern_updated(MbedCloudClientResource *resource, m2m::String newValue) {
    printf("PUT received, new value: %s\n", newValue.c_str());
}

/**
 * POST handler
 * @param resource The resource that triggered the callback
 * @param buffer If a body was passed to the POST function, this contains the data.
 *               Note that the buffer is deallocated after leaving this function, so copy it if you need it longer.
 * @param size Size of the body
 */
void blink_callback(MbedCloudClientResource *resource, const uint8_t *buffer, uint16_t size) {
    printf("POST received. Going to blink LED pattern: %s\n", pattern_res->get_value().c_str());

    static DigitalOut augmentedLed(LED1); // LED that is used for blinking the pattern

    // Parse the pattern string, and toggle the LED in that pattern
    string s = std::string(pattern_res->get_value().c_str());
    size_t i = 0;
    size_t pos = s.find(':');
    while (pos != string::npos) {
        wait_ms(atoi(s.substr(i, pos - i).c_str()));
        augmentedLed = !augmentedLed;

        i = ++pos;
        pos = s.find(':', pos);

        if (pos == string::npos) {
            wait_ms(atoi(s.substr(i, s.length()).c_str()));
            augmentedLed = !augmentedLed;
        }
    }
}

void camera_action_callback(MbedCloudClientResource *resource, const uint8_t *buffer, uint16_t size) {
    printf("POST received. Going to camera action\n");

    save_camera_image_req();
}

/**
 * Notification callback handler
 * @param resource The resource that triggered the callback
 * @param status The delivery status of the notification
 */
void button_callback(MbedCloudClientResource *resource, const NoticationDeliveryStatus status) {
    printf("Button notification, status %s (%d)\n", MbedCloudClientResource::delivery_status_to_string(status), status);
}

void camera_img_callback(MbedCloudClientResource *resource, const NoticationDeliveryStatus status) {
    printf("Camera image notification, status %s (%d)\n", MbedCloudClientResource::delivery_status_to_string(status), status);
}

/**
 * Registration callback handler
 * @param endpoint Information about the registered endpoint such as the name (so you can find it back in portal)
 */
void registered(const ConnectorClientEndpointInfo *endpoint) {
    printf("Connected to Pelion Device Management. Endpoint Name: %s\n", endpoint->internal_endpoint_name.c_str());
}

int main(void) {
    printf("Starting Simple Pelion Device Management Client example\n");
    printf("Connecting to the network...\n");

    // Connect to the internet (DHCP is expected to be on)
    nsapi_error_t status = net->connect();

    if (status != NSAPI_ERROR_OK) {
        printf("Connecting to the network failed %d!\n", status);
        return -1;
    }

    printf("Connected to the network successfully. IP address: %s\n", net->get_ip_address());

    // SimpleMbedCloudClient handles registering over LwM2M to Pelion Device Management
    SimpleMbedCloudClient client(net, bd, &fs);
    int client_status = client.init();
    if (client_status != 0) {
        printf("Pelion Client initialization failed (%d)\n", client_status);
        return -1;
    }

    // Creating resources, which can be written or read from the cloud
    button_res = client.create_resource("3200/0/5501", "button_count");
    button_res->set_value(0);
    button_res->methods(M2MMethod::GET);
    button_res->observable(true);
    button_res->attach_notification_callback(button_callback);

    pattern_res = client.create_resource("3201/0/5853", "blink_pattern");
    pattern_res->set_value("500:500:500:500:500:500:500:500");
    pattern_res->methods(M2MMethod::GET | M2MMethod::PUT);
    pattern_res->attach_put_callback(pattern_updated);

    MbedCloudClientResource *blink_res = client.create_resource("3201/0/5850", "blink_action");
    blink_res->methods(M2MMethod::POST);
    blink_res->attach_post_callback(blink_callback);

    MbedCloudClientResource *camera_img_res = client.create_resource("5000/0/1", "camera_img");
    camera_img_res->set_value(0);
    camera_img_res->methods(M2MMethod::GET);
    camera_img_res->observable(true);
    camera_img_res->attach_notification_callback(camera_img_callback);

    MbedCloudClientResource *camera_action_res = client.create_resource("5000/0/3", "camera_action");
    camera_action_res->methods(M2MMethod::POST);
    camera_action_res->attach_post_callback(camera_action_callback);

    printf("Initialized Pelion Client. Registering...\n");

    // Callback that fires when registering is complete
    client.on_registered(&registered);

    // Register with Pelion Device Management
    client.register_and_connect();

    // Setup the button
    btn.mode(PullUp);

    // The button fall handler is placed in the event queue so it will run in
    // thread context instead of ISR context, which allows safely updating the cloud resource
    btn.fall(eventQueue.event(&button_press));

    // Start jpeg task
    jpegTask.start(callback(jpeg_task, camera_img_res));

    // You can easily run the eventQueue in a separate thread if required
    eventQueue.dispatch_forever();
}
#endif
