/* FTP client library
 * Copyright (c) 2018 Renesas Electronics Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef FTP_CLIENT_H
#define FTP_CLIENT_H
#include "mbed.h"

/** FTPClient class.
 */
class FTPClient {
public:
    /** Constructor
     *
     * @param net     FTP server port
     * @param root    User name
     */
    FTPClient(NetworkInterface *net, const char* root);
    ~FTPClient();

    /** Opens address.
     *
     * @param ip_addr FTP server IP address
     * @param port    FTP server port
     * @param user    User name
     * @param pass    Password
     * @return true = success, false = failure
     */
    bool open(const char* ip_addr, int port, const char* user, const char* pass);

    /** Exits from FTP.
     *
     * @return true = success, false = failure
     */
    bool quit();

    /** Get file from the remote computer.
     *
     * @param file_name File name
     * @return true = success, false = failure
     */
    bool get(const char* file_name);

    /** Send one file.
     *
     * @param file_name File name
     * @return true = success, false = failure
     */
    bool put(const char* file_name);

    /** Deletes a file.
     *
     * @param file_name File name
     * @return true = success, false = failure
     */
    bool del(const char* file_name);

    /** Make directory.
     *
     * @param dir_name  Directory name
     * @return true = success, false = failure
     */
    bool mkdir(const char* dir_name);

    /** Changes directory.
     *
     * @param dir_name  Directory name
     * @return true = success, false = failure
     */
    bool cd(const char* dir_name);

    /** Lists files, if connected.
     *
     * @param list_buf  Buffer to store results
     * @param buf_size  Size of list_buf
     * @return true = success, false = failure
     */
    bool dir(char* list_buf, int buf_size);

    /** Lists files of the remotely connected computer.
     *
     * @param list_buf  Buffer to store results
     * @param buf_size  Size of list_buf
     * @return true = success, false = failure
     */
    bool ls(char* list_buf, int buf_size);

private:
    NetworkInterface * p_network;
    TCPSocket FTPClientControlSock;
    TCPSocket FTPClientDataSock;
    bool _ctr_open;
    bool _login;
    char ftp_data_ip_addr[20];
    char * p_ftp_buf;
    char _root[20];

    bool open_data_sock();
};
#endif
