/* mbed Microcontroller Library
 * Copyright (C) 2016 Renesas Electronics Corporation. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "MotionEventConverter.h"

#define MULTI_TOUCH_NUM         (2)

MotionEventConverter::MotionEventConverter() : sem_touch_int(0) {
}

void MotionEventConverter::touch_process(TouchKey * p_touch) {
    memset(touch_pos, 0, sizeof(touch_pos));
    memset(touch_pos_last, 0, sizeof(touch_pos_last));
    time_cnt = 0;
    touch_num = 0;
    touch_num_last = 0;

    p_touch->SetCallback(this, &MotionEventConverter::touch_int_callback);
    p_touch->Reset();
    t.reset();
    t.start();

    while (1) {
        sem_touch_int.wait();
        // Get Coordinates
        touch_num = p_touch->GetCoordinates(MotionEvent::TOUCH_NUM_MAX, &touch_pos[0]);
        time_cnt = t.read_ms();
        event.clearPointerCount();
        if ((touch_num_last == 0) && (touch_num >= MULTI_TOUCH_NUM)) {
            // Two points are pressed at the same time
            event.setPosData(0, touch_pos[0].x, touch_pos[0].y);
            event.setActionInfo((0 << MotionEvent::ACTION_POINTER_INDEX_SHIFT) | ACTION_DOWN, time_cnt);
            _callback.call(event);

            event.setPosData(1, touch_pos[1].x, touch_pos[1].y);
            event.setActionInfo((1 << MotionEvent::ACTION_POINTER_INDEX_SHIFT) | ACTION_POINTER_DOWN, time_cnt);
            _callback.call(event);
        } else if ((touch_num_last == 0) && (touch_num == 1)) {
            // It was pressed only one point
            if (touch_pos[0].valid != false) {
                pidx = 0;
            } else {
                pidx = 1;
            }
            event.setPosData(pidx, touch_pos[pidx].x, touch_pos[pidx].y);
            event.setActionInfo((0 << MotionEvent::ACTION_POINTER_INDEX_SHIFT) | ACTION_DOWN, time_cnt);
            _callback.call(event);
        } else if ((touch_num_last == 1) && (touch_num == 2)) {
            // The second point is pressed
            event.setPosData(0, touch_pos[0].x, touch_pos[0].y);
            event.setPosData(1, touch_pos[1].x, touch_pos[1].y);
            if (touch_pos[0].valid != touch_pos_last[0].valid) {
                pidx = 0;
            } else {
                pidx = 1;
            }
            event.setActionInfo((pidx << MotionEvent::ACTION_POINTER_INDEX_SHIFT) | ACTION_POINTER_DOWN, time_cnt);
            _callback.call(event);
        } else if ((touch_num_last >= MULTI_TOUCH_NUM) && (touch_num == 0)) {
            // Two points are released at the same time
            event.setPosData(0, touch_pos_last[0].x, touch_pos_last[0].y);
            event.setPosData(1, touch_pos_last[1].x, touch_pos_last[1].y);
            event.setActionInfo((1 << MotionEvent::ACTION_POINTER_INDEX_SHIFT) | ACTION_POINTER_UP, time_cnt);
            _callback.call(event);

            event.clearPointerCount();
            event.setPosData(0, touch_pos_last[0].x, touch_pos_last[0].y);
            event.setActionInfo((0 << MotionEvent::ACTION_POINTER_INDEX_SHIFT) | ACTION_UP, time_cnt);
            _callback.call(event);
        } else if ((touch_num_last == 1) && (touch_num == 0)) {
            // The last of one point is released
            if (touch_pos[0].valid != touch_pos_last[0].valid) {
                pidx = 0;
            } else {
                pidx = 1;
            }
            event.setPosData(pidx, touch_pos_last[pidx].x, touch_pos_last[pidx].y);
            event.setActionInfo((0 << MotionEvent::ACTION_POINTER_INDEX_SHIFT) | ACTION_UP, time_cnt);
            _callback.call(event);
        } else if ((touch_num_last >= MULTI_TOUCH_NUM) && (touch_num == 1)) {
            // The second point is released
            pidx = 0;
            if (touch_pos[0].valid != false) {
                event.setPosData(0, touch_pos[0].x, touch_pos[0].y);
            } else {
                event.setPosData(0, touch_pos_last[0].x, touch_pos_last[0].y);
            }
            if (touch_pos[1].valid != false) {
                event.setPosData(1, touch_pos[1].x, touch_pos[1].y);
            } else {
                event.setPosData(1, touch_pos_last[1].x, touch_pos_last[1].y);
                pidx = 1;
            }
            event.setActionInfo((pidx << MotionEvent::ACTION_POINTER_INDEX_SHIFT) | ACTION_POINTER_UP, time_cnt);
            _callback.call(event);
        } else if (touch_num != 0) {
            // Check whether the position has moved
            bool moved = false;
            for (int i = 0; i < MotionEvent::TOUCH_NUM_MAX; i++) {
                if (touch_pos[i].valid != false) {
                    event.setPosData(i, touch_pos[i].x, touch_pos[i].y);
                    if ((touch_pos[i].x != touch_pos_last[i].x) || (touch_pos[i].y != touch_pos_last[i].y)) {
                        moved = true;
                        pidx  = i;
                    }
                }
            }
            if (moved != false) {
                if (touch_num >= MULTI_TOUCH_NUM) {
                    pidx = 0;
                }
                event.setActionInfo((pidx << MotionEvent::ACTION_POINTER_INDEX_SHIFT) | ACTION_MOVE, time_cnt);
                _callback.call(event);
            }
        } else {
            // do nothing
        }
        touch_pos_last[0] = touch_pos[0];
        touch_pos_last[1] = touch_pos[1];
        touch_num_last = touch_num;
    }
}

void MotionEventConverter::touch_int_callback(void) {
    sem_touch_int.release();
}



