/* mbed Microcontroller Library
 * Copyright (C) 2016 Renesas Electronics Corporation. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**************************************************************************//**
* @file          MotionEventConverter.h
* @brief         MotionEventConverter API
******************************************************************************/

#ifndef MOTION_EVENT_CONVERTER_H
#define MOTION_EVENT_CONVERTER_H

#include "mbed.h"
#include "rtos.h"
#include "MotionEvent.h"
#include "TouchKey.h"

/**
 * A class to communicate a MotionEventConverter
 */
class MotionEventConverter {

public:
    /** Constructor
     *
     */
    MotionEventConverter();

    /** Run the touch panel process.
     *
     *  @param fptr A pointer to a void function
     *  @param p_touch pointer of TouchKey class
     */
    void Process(int (*fptr)(MotionEvent event), TouchKey * p_touch) {
        _callback.attach(fptr);
        touch_process(p_touch);
    }

    /** Run the touch panel process.
     *
     *  @param tptr pointer to the object to call the member function on
     *  @param mptr pointer to the member function to be called
     *  @param p_touch pointer of TouchKey class
     */
    template<typename T>
    void Process(T* tptr, int (T::*mptr)(MotionEvent event), TouchKey * p_touch) {
        _callback.attach(tptr, mptr);
        touch_process(p_touch);
    }

private:
    TouchKey *            p_touch;
    MotionEventCtl        event;
    Timer                 t;
    Semaphore             sem_touch_int;
    TouchKey::touch_pos_t touch_pos[MotionEvent::TOUCH_NUM_MAX];
    TouchKey::touch_pos_t touch_pos_last[MotionEvent::TOUCH_NUM_MAX];
    uint32_t              time_cnt;
    int                   pidx;
    uint8_t               touch_num;
    uint8_t               touch_num_last;
    FunctionPointerArg1<int, MotionEvent>  _callback;

    void touch_process(TouchKey * p_touch);
    void touch_int_callback(void);
};

#endif
