/* mbed Microcontroller Library
 * Copyright (C) 2016 Renesas Electronics Corporation. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**************************************************************************//**
* @file          MotionEvent.h
* @brief         MotionEvent API
******************************************************************************/

#ifndef MOTION_EVENT_H
#define MOTION_EVENT_H

#include "mbed.h"

#define ACTION_DOWN            (0x00000000)
#define ACTION_UP              (0x00000001)
#define ACTION_MOVE            (0x00000002)
#define ACTION_CANCEL          (0x00000003)
#define ACTION_POINTER_DOWN    (0x00000005)
#define ACTION_POINTER_UP      (0x00000006)

/**
 * MotionEvent
 */
class MotionEvent {

public:
    /** getX(int) for the first pointer index (may be an arbitrary pointer identifier). 
     *
     * @return float
     */
    float getX();

    /** Returns the X coordinate of this event for the given pointer index 
     * (use getPointerId(int) to find the pointer identifier for this index).
     * Whole numbers are pixels; the value may have a fraction for input devices that are sub-pixel precise.
     *
     * @param pointerIndex Raw index of pointer to retrieve.
     *        Value may be from 0 (the first pointer that is down) to getPointerCount()-1.
     * @return float
     */
    float getX(int pointerIndex);

    /** getY(int) for the first pointer index (may be an arbitrary pointer identifier).
     *
     * @return float
     */
    float getY();

    /** Returns the Y coordinate of this event for the given pointer index 
     * (use getPointerId(int) to find the pointer identifier for this index).
     * Whole numbers are pixels; the value may have a fraction for input devices that are sub-pixel precise.
     *
     * @param pointerIndex Raw index of pointer to retrieve.
     *        Value may be from 0 (the first pointer that is down) to getPointerCount()-1.
     * @return float
     */
    float getY(int pointerIndex);

    /** The number of pointers of data contained in this event. Always >= 1.
     *
     * @return int
     */
    int getPointerCount();

    /** Return the pointer identifier associated with a particular pointer data index in this event.
     * The identifier tells you the actual pointer number associated with the data, accounting for
     * individual pointers going up and down since the start of the current gesture.
     *
     * @return int
     */
    int getPointerId(int pointerIndex);

    /** Return the kind of action being performed. Consider using getActionMasked() and getActionIndex()
     *  to retrieve the separate masked action and pointer index.
     *
     * @return The action, such as ACTION_DOWN or the combination of ACTION_POINTER_DOWN with a shifted pointer index.
     */
    int getAction();

    /** Return the masked action being performed, without pointer index information. Use getActionIndex()
     *   to return the index associated with pointer actions.
     *
     * @return The action, such as ACTION_DOWN or ACTION_POINTER_DOWN.
     */
    int getActionMasked();

    /** Returns the time (in ms) when the user originally pressed down to start a stream of position events.
     *
     * @return long
     */
    long getDownTime();

    /** Retrieve the time this event occurred, in the uptimeMillis() time base.
     *
     * @return Returns the time this event occurred, in the uptimeMillis() time base.
     */
    long getEventTime();

    /** Given a pointer identifier, find the index of its data in the event.
     *
     * @param pointerId The identifier of the pointer to be found.
     * @return Returns either the index of the pointer (for use with getX(int) et al.),
     *  or -1 if there is no data available for that pointer identifier.
     */
    int findPointerIndex(int pointerId);


    /** Bits in the action code that represent a pointer index, used with ACTION_POINTER_DOWN and ACTION_POINTER_UP.
     * Shifting down by ACTION_POINTER_INDEX_SHIFT provides the actual pointer index where the data for the pointer
     * going up or down can be found; you can get its identifier with getPointerId(int) and the actual data with
     * getX(int) etc.
     */
    static const int ACTION_POINTER_INDEX_MASK  = 0x0000ff00;

    /** Bit shift for the action bits holding the pointer index as defined by ACTION_POINTER_INDEX_MASK.
     *
     */
    static const int ACTION_POINTER_INDEX_SHIFT = 0x00000008;

    /** Bit mask of the parts of the action code that are the action itself.
     *
     */
    static const int ACTION_MASK                = 0x000000ff;


    /** printf output function for debugging
     *
     */
    void debug_print();

    /** The maximum number of touch
     *
     */
    static const int TOUCH_NUM_MAX  = 2;

protected:
    typedef struct {
        uint32_t id;
        uint32_t x;
        uint32_t y;
    } touch_t;

    touch_t _s_touch[TOUCH_NUM_MAX];
    int     _action;
    long    _event_time;
    long    _down_time;
    int     _touch_idx;
};


/**
 * The class to set the price as MotionEvent
 */
class MotionEventCtl : public MotionEvent {

public:
    /** Initialize of touch information.
     *
     */
    void clearPointerCount();

    /** Set the coordinate.
     *
     * @param id PointerId.
     * @param x X coordinate.
     * @param y Y coordinate.
     */
    void setPosData(int id, uint32_t x, uint32_t y);

    /** Set the action.
     *
     * @param action The action, such as ACTION_DOWN or the combination of ACTION_POINTER_DOWN with a shifted pointer index.
     * @param time The time this event occurred, in the uptimeMillis() time base.
     */
    void setActionInfo(int action, long time);
};

#endif

