/* LWIP implementation of NetworkInterfaceAPI
 * Copyright (c) 2015 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "LWIPBP3595Interface.h"
#include "LWIPBP3595Interface_BssType.h"
#include "lwip_wifi_stack.h"
#include "WlanBP3595.h"

static bool _init_end = false;
static grp_wld_site_survey_result_array survey_result;

static void _wlan_inf_callback(uint8_t ucType, uint16_t usWid, uint16_t usSize, uint8_t *pucData) {
    if ((ucType == 'I') && (usWid == 0x0005)) {
        if (pucData[0] == 0x01) {     // CONNECTED
            /* Notify the LWIPBP3595Interface driver that WLAN was connected */
            WlanBP3595_Connected();
        } else {
            /* Notify the LWIPBP3595Interface driver that WLAN was disconnected */
            WlanBP3595_Disconnected();
        }
    }
}

static int _wlan_init() {
    uint32_t status;
    grp_u8  ucWidData8;     // 8bit wid data

    if (_init_end == false) {
        // Initialize WlanBP3595
        if (WlanBP3595_Init(&_wlan_inf_callback) != 0) {
            return -1;
        }

        // Wait until WLAN_BP3595_START  timeout 60s
        while (1) {
            Thread::wait(200);
            status = WlanBP3595_GetWlanSts();
            if (status == WLAN_BP3595_START) {
                break;
            }
        }

        // Set BSS type
        ucWidData8 = BSS_TYPE;
        if (WlanBP3595_Ioctl(GRP_WLD_IOCTL_SET_BSS_TYPE, &ucWidData8) != 0) {
            return -1;
        }

        ucWidData8 = 0x02;  // Diversity
        WlanBP3595_Ioctl(GRP_WLD_IOCTL_SET_SEL_ANTENNA, &ucWidData8);
        _init_end = true;
    }

    return 0;
}

static int _wlan_setting(const char *ssid, const char *pass, nsapi_security_t security)
{
    int     ret;
    grp_u8  ucWidData8;     // 8bit wid data
    grp_wld_byte_array  tBAWidData;     // byte array wid data

    // Set SSID
    tBAWidData.pucData = (grp_u8 *)ssid;
    tBAWidData.ulSize  = strlen((char *)tBAWidData.pucData);
    ret = WlanBP3595_Ioctl(GRP_WLD_IOCTL_SET_SSID, &tBAWidData);
    if (ret != 0) {
        return NSAPI_ERROR_AUTH_FAILURE;
    }

    if ((security == NSAPI_SECURITY_WPA)
     || (security == NSAPI_SECURITY_WPA2)
     || (security == NSAPI_SECURITY_WPA_WPA2)) {
        // Set PSK
        tBAWidData.pucData = (grp_u8 *)pass;
        tBAWidData.ulSize  = strlen((char *)tBAWidData.pucData);
        ret = WlanBP3595_Ioctl(GRP_WLD_IOCTL_SET_11I_PSK, &tBAWidData);
        if (ret != 0) {
            return NSAPI_ERROR_AUTH_FAILURE;
        }
    }

    // Set 11i mode
    switch (security) {
        case NSAPI_SECURITY_WEP:
            ret = strlen(pass);
            if (ret == 5) {
                ucWidData8 = 0x03;  // WEP64
            } else if (ret == 13) {
                ucWidData8 = 0x07;  // WEP128
            } else {
                return NSAPI_ERROR_PARAMETER;
            }
            break;
        case NSAPI_SECURITY_WPA:
            ucWidData8 = 0x69;  // WPA-TKIP/AES(PSK)
            break;
        case NSAPI_SECURITY_WPA2:
            ucWidData8 = 0x71;  // WPA2-TKIP/AES(PSK)
            break;
        case NSAPI_SECURITY_WPA_WPA2:
            ucWidData8 = 0x79;  // WPA/WPA2 Mixed
            break;
        case NSAPI_SECURITY_NONE:
        default:
            ucWidData8 = 0x00;
            break;
    }
    ret = WlanBP3595_Ioctl(GRP_WLD_IOCTL_SET_11I_MODE, &ucWidData8);
    if (ret != 0) {
        return NSAPI_ERROR_AUTH_FAILURE;
    }

    if (security == NSAPI_SECURITY_WEP) {
        // Set WEP KEY
        tBAWidData.pucData = (grp_u8 *)pass;
        tBAWidData.ulSize  = strlen((char *)tBAWidData.pucData);
        ret = WlanBP3595_Ioctl(GRP_WLD_IOCTL_SET_WEP_KEY, &tBAWidData);
        if (ret != 0) {
            return NSAPI_ERROR_AUTH_FAILURE;
        }
    }

    return 0;
}

/* Interface implementation */
LWIPBP3595Interface::LWIPBP3595Interface()
    : _dhcp(true), _ip_address(), _netmask(), _gateway()
{
}

nsapi_error_t LWIPBP3595Interface::set_network(const char *ip_address, const char *netmask, const char *gateway)
{
    _dhcp = false;
    strncpy(_ip_address, ip_address ? ip_address : "", sizeof(_ip_address));
    strncpy(_netmask, netmask ? netmask : "", sizeof(_netmask));
    strncpy(_gateway, gateway ? gateway : "", sizeof(_gateway));
    return NSAPI_ERROR_OK;
}

nsapi_error_t LWIPBP3595Interface::set_dhcp(bool dhcp)
{
    _dhcp = dhcp;
    return NSAPI_ERROR_OK;
}

nsapi_error_t LWIPBP3595Interface::set_credentials(const char *ssid, const char *pass, nsapi_security_t security)
{
    memset(_ssid, 0, sizeof(_ssid));
    strncpy(_ssid, ssid, sizeof(_ssid));

    memset(_pass, 0, sizeof(_pass));
    strncpy(_pass, pass, sizeof(_pass));

    _security = security;

    return 0;
}

nsapi_error_t LWIPBP3595Interface::set_channel(uint8_t channel)
{
    int     ret;
    grp_u8  ucWidData8;     // 8bit wid data

    if (_wlan_init() != 0) {
        return NSAPI_ERROR_DEVICE_ERROR;
    }

    if (channel != 0) {
        ucWidData8 = channel;
        ret = WlanBP3595_Ioctl(GRP_WLD_IOCTL_SET_CHANNEL, &ucWidData8);
        if (ret != 0) {
            return NSAPI_ERROR_PARAMETER;
        }
    }

    return 0;
}

int8_t LWIPBP3595Interface::get_rssi()
{
    int     ret;
    grp_u8  ucWidData8;     // 8bit wid data

    if (_init_end) {
        ret = WlanBP3595_Ioctl(GRP_WLD_IOCTL_GET_RSSI, &ucWidData8);
        if (ret != 0) {
            return 0;
        }

        return ucWidData8;
    }
    return 0;
}

nsapi_error_t LWIPBP3595Interface::connect(const char *ssid, const char *pass, nsapi_security_t security, uint8_t channel)
{
    set_credentials(ssid, pass, security);
    set_channel(channel);
    return connect();
}

nsapi_error_t LWIPBP3595Interface::connect()
{
    int ret;

    if (_wlan_init() != 0) {
        return NSAPI_ERROR_DEVICE_ERROR;
    }

    if (mbed_lwip_wifi_init(NULL) != NSAPI_ERROR_OK) {
        return NSAPI_ERROR_DEVICE_ERROR;
    }

    ret = _wlan_setting(_ssid, _pass, _security);
    if (ret != 0) {
        return ret;
    }

    return mbed_lwip_wifi_bringup(_dhcp,
            _ip_address[0] ? _ip_address : 0,
            _netmask[0] ? _netmask : 0,
            _gateway[0] ? _gateway : 0);
}

nsapi_error_t LWIPBP3595Interface::disconnect()
{
    return mbed_lwip_wifi_bringdown();
}

nsapi_size_or_error_t LWIPBP3595Interface::scan(WiFiAccessPoint *res, unsigned count)
{
    int     cnt;
    int     ret;
    grp_u8  ucWidData8;     // 8bit wid data
    nsapi_wifi_ap_t ap;

    if (_wlan_init() != 0) {
        return NSAPI_ERROR_DEVICE_ERROR;
    }

    ucWidData8 = 0x01;      // All channel
    ret = WlanBP3595_Ioctl(GRP_WLD_IOCTL_SET_SITE_SURVEY, &ucWidData8);
    if (ret != 0) {
        return NSAPI_ERROR_DEVICE_ERROR;
    }

    ucWidData8 = 0x01;
    ret = WlanBP3595_Ioctl(GRP_WLD_IOCTL_SET_START_SCAN_REQ, &ucWidData8);
    if (ret != 0) {
        return NSAPI_ERROR_DEVICE_ERROR;
    }

    while (1) {
        ucWidData8 = 0x00;
        ret = WlanBP3595_Ioctl(GRP_WLD_IOCTL_GET_CHECK_SCAN_END, &ucWidData8);
        if (ret != 0) {
            return NSAPI_ERROR_DEVICE_ERROR;
        }
        if (ucWidData8 == 0) {
            break;
        }
        Thread::wait(100);
    }

    ret = WlanBP3595_Ioctl(GRP_WLD_IOCTL_GET_SITE_SURVEY_RESULT, &survey_result);
    if (ret != 0) {
        return NSAPI_ERROR_DEVICE_ERROR;
    }

    for (cnt = 0; (cnt < count) && (cnt < survey_result.iCnt); cnt++) {
        grp_wld_site_survey_result * wk_result = &survey_result.atResult[cnt];

        memcpy(ap.ssid, wk_result->aucSsid, 33);
        memcpy(ap.bssid, wk_result->aucBssid, 6);
        switch (wk_result->ucSecurity) {
            case 0x00:
                ap.security = NSAPI_SECURITY_NONE;
                break;
            case 0x01:
                ap.security = NSAPI_SECURITY_WEP;
                break;
            case 0x29:
            case 0x49:
            case 0x69:
                ap.security = NSAPI_SECURITY_WPA;
                break;
            case 0x31:
            case 0x51:
            case 0x71:
                ap.security = NSAPI_SECURITY_WPA2;
                break;
            case 0x79:
                ap.security = NSAPI_SECURITY_WPA_WPA2;
                break;
            default:
                ap.security = NSAPI_SECURITY_UNKNOWN;
                break;
        }
        ap.rssi     = wk_result->ucRxPower;
        ap.channel  = wk_result->ucChannel;

        res[cnt] = WiFiAccessPoint(ap);
    }

    return cnt;
}

const char *LWIPBP3595Interface::get_mac_address()
{
    return mbed_lwip_wifi_get_mac_address();
}

const char *LWIPBP3595Interface::get_ip_address()
{
    if (mbed_lwip_wifi_get_ip_address(_ip_address, sizeof _ip_address)) {
        return _ip_address;
    }

    return NULL;
}

const char *LWIPBP3595Interface::get_netmask()
{
    if (mbed_lwip_wifi_get_netmask(_netmask, sizeof _netmask)) {
        return _netmask;
    }

    return NULL;
}

const char *LWIPBP3595Interface::get_gateway()
{
    if (mbed_lwip_wifi_get_gateway(_gateway, sizeof _gateway)) {
        return _gateway;
    }

    return NULL;
}

NetworkStack *LWIPBP3595Interface::get_stack()
{
    return nsapi_create_stack(&lwip_wifi_stack);
}

