#include "mbed.h"
#include "dcache-control.h"
#include "EasyAttach_CameraAndLCD.h"
#include "AsciiFont.h"

#define STRING_DISP_TEST       (1)

#if (MBED_CONF_APP_LCD_TYPE == GR_PEACH_4_3INCH_SHIELD)
#define ASPECT_RATIO_16_9      (1)
#endif

static DisplayBase Display;

/************************** Camera **************************/
#if MBED_CONF_APP_CAMERA
/*! Frame buffer stride: Frame buffer stride should be set to a multiple of 32 or 128
    in accordance with the frame buffer burst transfer mode. */
#define DATA_SIZE_PER_PIC      (2u)
#define FRAME_BUFFER_STRIDE    (((LCD_PIXEL_WIDTH * DATA_SIZE_PER_PIC) + 31u) & ~31u)
#define FRAME_BUFFER_HEIGHT    (LCD_PIXEL_HEIGHT)

#if defined(__ICCARM__)
#pragma data_alignment=32
static uint8_t user_frame_buffer0[FRAME_BUFFER_STRIDE * FRAME_BUFFER_HEIGHT]@ ".mirrorram";
#else
static uint8_t user_frame_buffer0[FRAME_BUFFER_STRIDE * FRAME_BUFFER_HEIGHT]__attribute((section("NC_BSS"),aligned(32)));
#endif

static void Start_Video_Camera(void) {
    DisplayBase::rect_t rect;

    // Initialize the background to black
    for (uint32_t i = 0; i < sizeof(user_frame_buffer0); i += 2) {
        user_frame_buffer0[i + 0] = 0x00;
        user_frame_buffer0[i + 1] = 0x80;
    }

    // Video capture setting (progressive form fixed)
    Display.Video_Write_Setting(
        DisplayBase::VIDEO_INPUT_CHANNEL_0,
        DisplayBase::COL_SYS_NTSC_358,
        (void *)user_frame_buffer0,
        FRAME_BUFFER_STRIDE,
        DisplayBase::VIDEO_FORMAT_YCBCR422,
        DisplayBase::WR_RD_WRSWA_32_16BIT,
        LCD_PIXEL_HEIGHT,
        LCD_PIXEL_WIDTH
    );
    EasyAttach_CameraStart(Display, DisplayBase::VIDEO_INPUT_CHANNEL_0);

    // LCD setting
    rect.vs = 0;
    rect.vw = LCD_PIXEL_HEIGHT;
    rect.hs = 0;
    rect.hw = LCD_PIXEL_WIDTH;
    Display.Graphics_Read_Setting(
        DisplayBase::GRAPHICS_LAYER_0,
        (void *)user_frame_buffer0,
        FRAME_BUFFER_STRIDE,
        DisplayBase::GRAPHICS_FORMAT_YCBCR422,
        DisplayBase::WR_RD_WRSWA_32_16BIT,
        &rect
    );
    Display.Graphics_Start(DisplayBase::GRAPHICS_LAYER_0);
}
#endif // MBED_CONF_APP_CAMERA


/************************** Touch panel **************************/
#if defined(TouckKey_LCD_shield)
/* TOUCH BUFFER Parameter GRAPHICS_LAYER_1 */
#define TOUCH_BUFFER_BYTE_PER_PIXEL   (2u)
#define TOUCH_BUFFER_STRIDE           (((LCD_PIXEL_WIDTH * TOUCH_BUFFER_BYTE_PER_PIXEL) + 31u) & ~31u)
/* Touch panel parameter */
#define TOUCH_NUM                     (2u)
#define DRAW_POINT                    (8)

static Semaphore   sem_touch_int(0);
#if defined(__ICCARM__)
#pragma data_alignment=32
static uint8_t user_frame_buffer_touch[TOUCH_BUFFER_STRIDE * LCD_PIXEL_HEIGHT];
#else
static uint8_t user_frame_buffer_touch[TOUCH_BUFFER_STRIDE * LCD_PIXEL_HEIGHT]__attribute((aligned(32)));
#endif

static void draw_touch_pos(uint8_t * p_buf, int id, int x, int y) {
    int idx_base;
    int wk_idx;
    int i;
    int j;
    uint8_t coller_pix[TOUCH_BUFFER_BYTE_PER_PIXEL];  /* ARGB4444 */

    /* A coordinate in the upper left is calculated from a central coordinate. */
    if ((x - (DRAW_POINT / 2)) >= 0) {
        x -= (DRAW_POINT / 2);
    }
    if (x > ((int)LCD_PIXEL_WIDTH - DRAW_POINT)) {
        x = ((int)LCD_PIXEL_WIDTH - DRAW_POINT);
    }
    if ((y - (DRAW_POINT / 2)) >= 0) {
        y -= (DRAW_POINT / 2);
    }
    if (y > ((int)LCD_PIXEL_HEIGHT - DRAW_POINT)) {
        y = ((int)LCD_PIXEL_HEIGHT - DRAW_POINT);
    }
    idx_base = (x + ((int)LCD_PIXEL_WIDTH * y)) * TOUCH_BUFFER_BYTE_PER_PIXEL;

    /* Select color */
    if (id == 0) {
        /* Blue */
        coller_pix[0] = 0x0F;  /* 4:Green 4:Blue */
        coller_pix[1] = 0xF0;  /* 4:Alpha 4:Red  */
    } else {
        /* Pink */
        coller_pix[0] = 0x07;  /* 4:Green 4:Blue */
        coller_pix[1] = 0xFF;  /* 4:Alpha 4:Red  */
    }

    /* Drawing */
    for (i = 0; i < DRAW_POINT; i++) {
        wk_idx = idx_base + ((int)LCD_PIXEL_WIDTH * TOUCH_BUFFER_BYTE_PER_PIXEL * i);
        for (j = 0; j < DRAW_POINT; j++) {
            p_buf[wk_idx++] = coller_pix[0];
            p_buf[wk_idx++] = coller_pix[1];
        }
    }
}

static void touch_int_callback(void) {
    sem_touch_int.release();
}

static void touch_task(void) {
    DisplayBase::rect_t rect;
    TouchKey::touch_pos_t touch_pos[TOUCH_NUM];
    int touch_num = 0;
    int touch_num_last = 0;
    uint32_t i;
    TouckKey_LCD_shield touch(P4_0, P2_13, I2C_SDA, I2C_SCL);

    /* The layer by which the touch panel location is drawn */
    memset(user_frame_buffer_touch, 0, sizeof(user_frame_buffer_touch));
    dcache_clean(user_frame_buffer_touch, sizeof(user_frame_buffer_touch));
    rect.vs = 0;
    rect.vw = LCD_PIXEL_HEIGHT;
    rect.hs = 0;
    rect.hw = LCD_PIXEL_WIDTH;
    Display.Graphics_Read_Setting(
        DisplayBase::GRAPHICS_LAYER_1,
        (void *)user_frame_buffer_touch,
        TOUCH_BUFFER_STRIDE,
        DisplayBase::GRAPHICS_FORMAT_ARGB4444,
        DisplayBase::WR_RD_WRSWA_32_16BIT,
        &rect
    );
    Display.Graphics_Start(DisplayBase::GRAPHICS_LAYER_1);

    /* Callback setting */
    touch.SetCallback(&touch_int_callback);

    /* Reset touch IC */
    touch.Reset();

    while (1) {
        /* Wait touch event */
        sem_touch_int.wait();

        /* Get touch coordinates */
        touch_num = touch.GetCoordinates(TOUCH_NUM, touch_pos);

        /* When it's a new touch, touch frame buffer is initialized */
        if ((touch_num != 0) && (touch_num_last == 0)) {
            memset(user_frame_buffer_touch, 0, sizeof(user_frame_buffer_touch));
        }
        touch_num_last = touch_num;

        /* Drawing of a touch coordinate */
        for (i = 0; i < TOUCH_NUM; i ++) {
            printf("{valid=%d,x=%lu,y=%lu} ", touch_pos[i].valid, touch_pos[i].x, touch_pos[i].y);
            if (touch_pos[i].valid) {
                draw_touch_pos(user_frame_buffer_touch, i, touch_pos[i].x, touch_pos[i].y);
            }
        }
        printf("\r\n");

        /* Data cache clean */
        dcache_clean(user_frame_buffer_touch, sizeof(user_frame_buffer_touch));
    }
}
#endif // TouckKey_LCD_shield


/************************** String Disp Test **************************/
#if STRING_DISP_TEST
/* STRING BUFFER Parameter GRAPHICS_LAYER_2 */
#define STRING_PIXEL_HW               (120)
#define STRING_PIXEL_VM               (24)
#define STRING_BUFFER_BYTE_PER_PIXEL  (2u)
#define STRING_BUFFER_STRIDE          (((LCD_PIXEL_WIDTH * STRING_BUFFER_BYTE_PER_PIXEL) + 31u) & ~31u)

#if defined(__ICCARM__)
#pragma data_alignment=32
static uint8_t user_frame_buffer_string[STRING_BUFFER_STRIDE * STRING_PIXEL_VM];
#else
static uint8_t user_frame_buffer_string[STRING_BUFFER_STRIDE * STRING_PIXEL_VM]__attribute((aligned(32)));
#endif

static void string_task(void) {
    DisplayBase::rect_t rect;
    char test_cnt = 0x20;

    /* The layer by which the character string is drawn */
    memset(user_frame_buffer_string, 0, sizeof(user_frame_buffer_string));
    dcache_clean(user_frame_buffer_string, sizeof(user_frame_buffer_string));

    rect.vs = LCD_PIXEL_HEIGHT - STRING_PIXEL_VM - 10;
    rect.vw = STRING_PIXEL_VM;
    rect.hs = LCD_PIXEL_WIDTH - STRING_PIXEL_HW - 10;
    rect.hw = STRING_PIXEL_HW;
    Display.Graphics_Read_Setting(
        DisplayBase::GRAPHICS_LAYER_2,
        (void *)user_frame_buffer_string,
        STRING_BUFFER_STRIDE,
        DisplayBase::GRAPHICS_FORMAT_ARGB4444,
        DisplayBase::WR_RD_WRSWA_32_16BIT,
        &rect
    );
    Display.Graphics_Start(DisplayBase::GRAPHICS_LAYER_2);

    /* String */
    AsciiFont ascii_font(user_frame_buffer_string, STRING_PIXEL_HW, STRING_PIXEL_VM,
                         STRING_BUFFER_STRIDE, STRING_BUFFER_BYTE_PER_PIXEL);
    ascii_font.DrawStr("Font:", 0, 8, 0x0000ffff, 2);

    while (1) {
        //colour: rrrrGBAR (r:Reserve G:Green B:Blue A:Alpha R:Red
        ascii_font.DrawChar(test_cnt, 84, 0, 0x0000aa9f, 3);
        if (test_cnt < 0x7e) {
            test_cnt++;
        } else {
            test_cnt = 0x20;
        }
        dcache_clean(user_frame_buffer_string, sizeof(user_frame_buffer_string));
        ThisThread::sleep_for(1000);
    }
}
#endif // STRING_DISP_TEST


/************************** main **************************/
int main(void) {
    /* Camera and LCD setting */
#if ASPECT_RATIO_16_9
    EasyAttach_Init(Display, 640, 360);  //aspect ratio 16:9
#else
    EasyAttach_Init(Display);            //aspect ratio 4:3
#endif

    /* Start camera */
#if MBED_CONF_APP_CAMERA
    Start_Video_Camera();
#endif // MBED_CONF_APP_CAMERA

    /* LCD Backlight ON */
    ThisThread::sleep_for(50);  // After reset, wait a bit so that the power does not rise abruptly.
    EasyAttach_LcdBacklight(true);

    /* Start touch panel processing */
#if defined(TouckKey_LCD_shield)
    Thread touchTask;
    touchTask.start(callback(touch_task));
#endif // TouckKey_LCD_shield

#if STRING_DISP_TEST
    /* Start string disp processing */
    Thread stringTask;
    stringTask.start(callback(string_task));
#endif // STRING_DISP_TEST

    /* LED blink */
    DigitalOut  led_blue(LED_BLUE);
    while (1) {
        led_blue = !led_blue;
        ThisThread::sleep_for(500);
    }
}
