#include "mbed.h"
#include "DisplayBace.h"
#include "rtos.h"
#include "LCD_shield_config.h"
#if (LCD_TYPE == 0)
  #include "TouchKey_4_3inch.h"
  #define TouckKey_LCD_shield               TouchKey_4_3inch
#else
  #include "TouchKey_7_1inch.h"
  #define TouckKey_LCD_shield               TouchKey_7_1inch
#endif

/* FRAME BUFFER Parameter */
#define FRAME_BUFFER_BYTE_PER_PIXEL         (2)
#define FRAME_BUFFER_STRIDE                 (((LCD_PIXEL_WIDTH * FRAME_BUFFER_BYTE_PER_PIXEL) + 31u) & ~31u)

#define TOUCH_NUM                           (2)
#define DROW_POINT                          (3)

DisplayBase Display;
DigitalOut  lcd_pwon(P7_15);
DigitalOut  lcd_blon(P8_1);
PwmOut      lcd_cntrst(P8_15);
Serial      pc(USBTX, USBRX);
Semaphore   sem_touch_int(0);
TouckKey_LCD_shield touch(P4_0, P2_13, I2C_SDA, I2C_SCL);

static uint8_t user_frame_buffer[FRAME_BUFFER_STRIDE * LCD_PIXEL_HEIGHT]__attribute((section("NC_BSS"),aligned(32)));  /* 32 bytes aligned */
static volatile int32_t vsync_count = 0;

static void IntCallbackFunc_Vsync(DisplayBase::int_type_t int_type) {
    /* Interrupt callback function for Vsync interruption */
    if (vsync_count > 0) {
        vsync_count--;
    }
}

static void Wait_Vsync(const int32_t wait_count) {
    /* Wait for the specified number of times Vsync occurs */
    vsync_count = wait_count;
    while (vsync_count > 0) {
        /* Do nothing */
    }
}

static void Init_LCD_Display(void) {
    /* Create DisplayBase object */
    DisplayBase::graphics_error_t error;
    DisplayBase::lcd_config_t lcd_config;
    PinName lvds_pin[8] = {
        /* data pin */
        P5_7, P5_6, P5_5, P5_4, P5_3, P5_2, P5_1, P5_0
    };

    lcd_pwon = 0;
    lcd_blon = 0;
    Thread::wait(100);
    lcd_pwon = 1;
    lcd_blon = 1;
    Thread::wait(100);
    lcd_cntrst.write(1.0);

    /* Graphics initialization process */
    GetLcdConfig(&lcd_config);
    error = Display.Graphics_init(&lcd_config);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        while (1);
    }

    /* Interrupt callback function setting (Vsync signal output from scaler 0) */
    error = Display.Graphics_Irq_Handler_Set(DisplayBase::INT_TYPE_S0_LO_VSYNC, 0, IntCallbackFunc_Vsync);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        while (1);
    }

    Display.Graphics_Lvds_Port_Init(lvds_pin, 8);
}

static void drow_touch_pos(int id, int x, int y) {
    int idx_base;
    int wk_idx;
    int i;
    int j;
    uint8_t coller_pix[2];

    if ((x - (DROW_POINT / 2)) >= 0) {
        x -= (DROW_POINT / 2);
    }
    if (x > (LCD_PIXEL_WIDTH - DROW_POINT)) {
        x = (LCD_PIXEL_WIDTH - DROW_POINT);
    }
    if ((y - (DROW_POINT / 2)) >= 0) {
        y -= (DROW_POINT / 2);
    }
    if (y > (LCD_PIXEL_HEIGHT - DROW_POINT)) {
        y = (LCD_PIXEL_HEIGHT - DROW_POINT);
    }
    idx_base = (x + (LCD_PIXEL_WIDTH * y)) * FRAME_BUFFER_BYTE_PER_PIXEL;

    if (id == 0) {
        coller_pix[0] = 0xDD;
        coller_pix[1] = 0x9E;
    } else {
        coller_pix[0] = 0x34;
        coller_pix[1] = 0xDB;
    }

    for (i = 0; i < DROW_POINT; i++) {
        wk_idx = idx_base + (LCD_PIXEL_WIDTH * FRAME_BUFFER_BYTE_PER_PIXEL * i);
        for (j = 0; j < DROW_POINT; j++) {
            user_frame_buffer[wk_idx++] = coller_pix[0];
            user_frame_buffer[wk_idx++] = coller_pix[1];
        }
    }
}

static void drow_touch_keyonoff(int id, bool onoff) {
    int idx_base;
    int wk_idx;
    int i;
    int j;
    uint8_t coller_pix[2];

    if (id == 0) {
        idx_base = 0;
    } else {
        idx_base = 5 * FRAME_BUFFER_BYTE_PER_PIXEL;
    }

    if (onoff == false) {
        coller_pix[0] = 0x00;
        coller_pix[1] = 0x00;
    } else {
        coller_pix[0] = 0xff;
        coller_pix[1] = 0xff;
    }

    for (i = 0; i < 5; i++) {
        wk_idx = idx_base + (LCD_PIXEL_WIDTH * FRAME_BUFFER_BYTE_PER_PIXEL * i);
        for (j = 0; j < 5; j++) {
            user_frame_buffer[wk_idx++] = coller_pix[0];
            user_frame_buffer[wk_idx++] = coller_pix[1];
        }
    }
}

static void touch_int_callback(void) {
    sem_touch_int.release();
}

int main(void) {
    // Change the baud rate of the printf()
    pc.baud(921600);

    // Initialization of LCD
    DisplayBase::rect_t rect;

    Init_LCD_Display();
    memset(user_frame_buffer, 0, sizeof(user_frame_buffer));
    rect.vs = 0;
    rect.vw = LCD_PIXEL_HEIGHT;
    rect.hs = 0;
    rect.hw = LCD_PIXEL_WIDTH;
    Display.Graphics_Read_Setting(
        DisplayBase::GRAPHICS_LAYER_0,
        (void *)user_frame_buffer,
        FRAME_BUFFER_STRIDE,
        DisplayBase::GRAPHICS_FORMAT_RGB565,
        DisplayBase::WR_RD_WRSWA_32_16BIT,
        &rect
    );
    Wait_Vsync(2);
    Display.Graphics_Start(DisplayBase::GRAPHICS_LAYER_0);

    // Touch panel processing
    TouchKey::touch_pos_t touch_pos[TOUCH_NUM];
    int touch_num = 0;
    int touch_num_last = 0;
    int i;

    touch.SetCallback(&touch_int_callback);
    touch.Reset();
    while (1) {
        sem_touch_int.wait();
        touch_num = touch.GetCoordinates(TOUCH_NUM, touch_pos);
        if ((touch_num != 0) && (touch_num_last == 0)) {
            memset(user_frame_buffer, 0, sizeof(user_frame_buffer));
        }
        for (i = 0; i < TOUCH_NUM; i ++) {
            printf("{valid=%d,x=%d,y=%d} ", touch_pos[i].valid, touch_pos[i].x, touch_pos[i].y);
            drow_touch_keyonoff(i, touch_pos[i].valid);
            if (touch_pos[i].valid) {
                drow_touch_pos(i, touch_pos[i].x, touch_pos[i].y);
            }
        }
        printf("\n");
        touch_num_last = touch_num;
    }
}
