#include "mbed.h"
#include "DisplayBace.h"
#include "rtos.h"

#define VIDEO_CVBS             (0)                 /* Analog  Video Signal */
#define VIDEO_CMOS_CAMERA      (1)                 /* Digital Video Signal */
#define VIDEO_YCBCR422         (0)
#define VIDEO_RGB888           (1)
#define VIDEO_RGB565           (2)

/**** User Selection *********/
#define VIDEO_INPUT_METHOD     (VIDEO_CVBS)        /* Select  VIDEO_CVBS or VIDEO_CMOS_CAMERA                       */
#define VIDEO_INPUT_FORMAT     (VIDEO_YCBCR422)    /* Select  VIDEO_YCBCR422 or VIDEO_RGB888 or VIDEO_RGB565        */
#define USE_VIDEO_CH           (0)                 /* Select  0 or 1            If selecting VIDEO_CMOS_CAMERA, should be 0.)               */
#define VIDEO_PAL              (0)                 /* Select  0(NTSC) or 1(PAL) If selecting VIDEO_CVBS, this parameter is not referenced.) */
/*****************************/
 
/**** LCD Parameter **********/
#define LCD_DE_MODE            (0)
#define LCD_SYNC_MODE          (1)

#define LCD_DOT_CLOCK          (13.40f)     // 13.4MHz

#define LCD_H_WIDTH            (480u)
#define LCD_H_BACK_PORCH       (43u)
#define LCD_H_FRONT_PORCH      (52u)
#define LCD_H_SYNC_WIDTH       (41u)

#define LCD_V_WIDTH            (272u)
#define LCD_V_BACK_PORCH       (12u)
#define LCD_V_FRONT_PORCH      (2u)
#define LCD_V_SYNC_WIDTH       (10u)

#define LCD_MODE               (LCD_SYNC_MODE)

/*****************************/


#if USE_VIDEO_CH == (0)
#define VIDEO_INPUT_CH         (DisplayBase::VIDEO_INPUT_CHANNEL_0)
#define VIDEO_INT_TYPE         (DisplayBase::INT_TYPE_S0_VFIELD)
#else
#define VIDEO_INPUT_CH         (DisplayBase::VIDEO_INPUT_CHANNEL_1)
#define VIDEO_INT_TYPE         (DisplayBase::INT_TYPE_S1_VFIELD)
#endif

#if ( VIDEO_INPUT_FORMAT == VIDEO_YCBCR422 || VIDEO_INPUT_FORMAT == VIDEO_RGB565 )
#define DATA_SIZE_PER_PIC      (2u)
#else
#define DATA_SIZE_PER_PIC      (4u)
#endif

/*! Frame buffer stride: Frame buffer stride should be set to a multiple of 32 or 128
    in accordance with the frame buffer burst transfer mode. */
#define PIXEL_HW               (480u)  /* WQVGA */
#define PIXEL_VW               (272u)  /* WQVGA */

#define VIDEO_BUFFER_STRIDE    (((PIXEL_HW * DATA_SIZE_PER_PIC) + 31u) & ~31u)
#define VIDEO_BUFFER_HEIGHT    (PIXEL_VW)

DigitalIn  button(USER_BUTTON0);
#if(1) //lcd
DigitalOut lcd_pwon(P7_15);
DigitalOut lcd_blon(P8_1);
DigitalOut touch_reset(P4_0);
PwmOut     lcd_cntrst(P8_15);
I2C        mI2c(I2C_SDA, I2C_SCL);

typedef struct {
    uint8_t y_h: 3,
    reserved: 1,
    x_h: 3,
    valid: 1;
    uint8_t x_l;
    uint8_t y_l;
} xyz_data_t;

typedef struct {
    uint8_t fingers: 3,
    reserved: 5;
    uint8_t keys;
    xyz_data_t xyz_data[2];
} stx_report_data_t;

static int get_coordinates(uint8_t *count, uint32_t *x0, uint32_t *y0, uint32_t *x1, uint32_t *y1) {
    char buf[8];
    stx_report_data_t *pdata;
    int ret = -1;
    *count = 0; // Set point detected count to 0.

    if (mI2c.read((0x55 << 1), buf, sizeof(buf)) == 0) {
        pdata = (stx_report_data_t *)buf;
        if (pdata->fingers) {
            if (pdata->xyz_data[0].valid) {
                *x0 = (pdata->xyz_data[0].x_h << 8) | pdata->xyz_data[0].x_l;
                *y0 = (pdata->xyz_data[0].y_h << 8) | pdata->xyz_data[0].y_l;
                (*count)++;
            }
        }
        if (pdata->xyz_data[1].valid) {
            *x1 = (pdata->xyz_data[1].x_h << 8) | pdata->xyz_data[1].x_l;
            *y1 = (pdata->xyz_data[1].y_h << 8) | pdata->xyz_data[1].y_l;
            (*count)++;
        }
        ret = 0;
    }

    return ret;
}
#endif

#if defined(__ICCARM__)
#pragma data_alignment=16
static uint8_t FrameBuffer_Video[VIDEO_BUFFER_STRIDE * VIDEO_BUFFER_HEIGHT]@ ".mirrorram";  //16 bytes aligned!;
#pragma data_alignment=4
#else
static uint8_t FrameBuffer_Video[VIDEO_BUFFER_STRIDE * VIDEO_BUFFER_HEIGHT]__attribute((section("NC_BSS"),aligned(16)));  //16 bytes aligned!;
#endif
static volatile int32_t vsync_count = 0;
static volatile int32_t vfield_count = 1;
static int image_change = 0;

static void IntCallbackFunc_Vfield(DisplayBase::int_type_t int_type)
{
    //Interrupt callback function
    if (vfield_count != 0) {
        vfield_count = 0;
    } else {
        vfield_count = 1;
        image_change = 1;
    }
}

static void IntCallbackFunc_Vsync(DisplayBase::int_type_t int_type)
{
    //Interrupt callback function for Vsync interruption
    if (vsync_count > 0) {
        vsync_count--;
    }
}

static void WaitVsync(const int32_t wait_count)
{
    //Wait for the specified number of times Vsync occurs
    vsync_count = wait_count;
    while (vsync_count > 0) {
        /* Do nothing */
    }
}

int main(void)
{
    /* Create DisplayBase object */
    DisplayBase Display;
    DisplayBase::graphics_error_t error;

#if VIDEO_INPUT_METHOD == VIDEO_CMOS_CAMERA
    DisplayBase::video_ext_in_config_t ext_in_config;
    PinName cmos_camera_pin[11] = {
        /* data pin */
        P2_7, P2_6, P2_5, P2_4, P2_3, P2_2, P2_1, P2_0,
        /* control pin */
        P10_0,      /* DV0_CLK   */
        P1_0,       /* DV0_Vsync */
        P1_1        /* DV0_Hsync */
    };
#endif
#if(1) //lcd
    lcd_pwon = 0;
    lcd_blon = 0;
    touch_reset = 0;
    Thread::wait(100);
 
    lcd_pwon = 1;
    lcd_blon = 1;
    touch_reset = 1;
    Thread::wait(100);

    DisplayBase::lcd_config_t lcd_config;
    PinName lvds_pin[8] = {
        /* data pin */
        P5_7, P5_6, P5_5, P5_4, P5_3, P5_2, P5_1, P5_0
    };
    DisplayBase::rect_t rect;

    lcd_config.lcd_type             = DisplayBase::LCD_TYPE_LVDS;
    lcd_config.intputClock          = 66.67f;
    lcd_config.outputClock          = LCD_DOT_CLOCK;
    lcd_config.lcd_outformat        = DisplayBase::LCD_OUTFORMAT_RGB888;
    lcd_config.lcd_edge             = DisplayBase::EDGE_RISING;
#if(LCD_MODE) //SYNC Mode
    lcd_config.h_toatal_period      = (LCD_H_BACK_PORCH + LCD_H_WIDTH + LCD_H_FRONT_PORCH);
    lcd_config.v_toatal_period      = (LCD_V_BACK_PORCH + LCD_V_WIDTH + LCD_V_FRONT_PORCH);
 
    lcd_config.h_disp_widht         = (LCD_H_WIDTH);
    lcd_config.v_disp_widht         = (LCD_V_WIDTH);
    lcd_config.h_back_porch         = (LCD_H_BACK_PORCH);
    lcd_config.v_back_porch         = (LCD_V_BACK_PORCH);
 
    lcd_config.h_sync_port          = DisplayBase::LCD_TCON_PIN_2;
    lcd_config.h_sync_port_polarity = DisplayBase::SIG_POL_INVERTED;
    lcd_config.h_sync_width         = LCD_H_SYNC_WIDTH;
 
    lcd_config.v_sync_port          = DisplayBase::LCD_TCON_PIN_0;
    lcd_config.v_sync_port_polarity = DisplayBase::SIG_POL_INVERTED;
    lcd_config.v_sync_width         = LCD_V_SYNC_WIDTH;
 
    lcd_config.de_port              = DisplayBase::LCD_TCON_PIN_3;
    lcd_config.de_port_polarity     = DisplayBase::SIG_POL_NOT_INVERTED;
#else  //DE Mode
    lcd_config.h_toatal_period      = (LCD_H_WIDTH + 80u);
    lcd_config.v_toatal_period      = (LCD_V_WIDTH);

    lcd_config.h_disp_widht         = (LCD_H_WIDTH);
    lcd_config.v_disp_widht         = (LCD_V_WIDTH);
    lcd_config.h_back_porch         = (68u);
    lcd_config.v_back_porch         = (18u);

    lcd_config.h_sync_port          = DisplayBase::LCD_TCON_PIN_NON;
    lcd_config.h_sync_port_polarity = DisplayBase::SIG_POL_NOT_INVERTED;
    lcd_config.h_sync_width         = 0;

    lcd_config.v_sync_port          = DisplayBase::LCD_TCON_PIN_NON;
    lcd_config.v_sync_port_polarity = DisplayBase::SIG_POL_NOT_INVERTED;
    lcd_config.v_sync_width         = 0;

    lcd_config.de_port              = DisplayBase::LCD_TCON_PIN_3;
    lcd_config.de_port_polarity     = DisplayBase::SIG_POL_INVERTED;
#endif

    /* Graphics initialization process */
    error = Display.Graphics_init(&lcd_config);
#else
    /* Graphics initialization process */
    error = Display.Graphics_init(NULL);
#endif

    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        while (1);
    }

#if VIDEO_INPUT_METHOD == VIDEO_CVBS
    error = Display.Graphics_Video_init( DisplayBase::INPUT_SEL_VDEC, NULL);
    if( error != DisplayBase::GRAPHICS_OK ) {
        printf("Line %d, error %d\n", __LINE__, error);
        while(1);
    }

#elif VIDEO_INPUT_METHOD == VIDEO_CMOS_CAMERA
    /* MT9V111 camera input config */
    ext_in_config.inp_format     = DisplayBase::VIDEO_EXTIN_FORMAT_BT601; /* BT601 8bit YCbCr format */
    ext_in_config.inp_pxd_edge   = DisplayBase::EDGE_RISING;              /* Clock edge select for capturing data          */
    ext_in_config.inp_vs_edge    = DisplayBase::EDGE_RISING;              /* Clock edge select for capturing Vsync signals */
    ext_in_config.inp_hs_edge    = DisplayBase::EDGE_RISING;              /* Clock edge select for capturing Hsync signals */
    ext_in_config.inp_endian_on  = DisplayBase::OFF;                      /* External input bit endian change on/off       */
    ext_in_config.inp_swap_on    = DisplayBase::OFF;                      /* External input B/R signal swap on/off         */
    ext_in_config.inp_vs_inv     = DisplayBase::SIG_POL_NOT_INVERTED;     /* External input DV_VSYNC inversion control     */
    ext_in_config.inp_hs_inv     = DisplayBase::SIG_POL_INVERTED;         /* External input DV_HSYNC inversion control     */
    ext_in_config.inp_f525_625   = DisplayBase::EXTIN_LINE_525;           /* Number of lines for BT.656 external input */
    ext_in_config.inp_h_pos      = DisplayBase::EXTIN_H_POS_CRYCBY;       /* Y/Cb/Y/Cr data string start timing to Hsync reference */
    ext_in_config.cap_vs_pos     = 6;                                     /* Capture start position from Vsync */
    ext_in_config.cap_hs_pos     = 150;                                   /* Capture start position form Hsync */
    ext_in_config.cap_width      = 640;                                   /* Capture width  */
    ext_in_config.cap_height     = 468u;                                  /* Capture height Max 468[line]
                                                                             Due to CMOS(MT9V111) output signal timing and VDC5 specification */
    error = Display.Graphics_Video_init( DisplayBase::INPUT_SEL_EXT, &ext_in_config);
    if( error != DisplayBase::GRAPHICS_OK ) {
        printf("Line %d, error %d\n", __LINE__, error);
        while(1);
    }

    /* MT9V111 camera input port setting */
    error = Display.Graphics_Dvinput_Port_Init(cmos_camera_pin, 11);
    if( error != DisplayBase::GRAPHICS_OK ) {
        printf("Line %d, error %d\n", __LINE__, error);
        while (1);
    }
#endif

    /* Interrupt callback function setting (Vsync signal input to scaler 0) */
    error = Display.Graphics_Irq_Handler_Set(DisplayBase::INT_TYPE_S0_VI_VSYNC, 0, IntCallbackFunc_Vsync);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        while (1);
    }
    /* Video capture setting (progressive form fixed) */
    error = Display.Video_Write_Setting(
                VIDEO_INPUT_CH,
#if VIDEO_PAL == 0
                DisplayBase::COL_SYS_NTSC_358,
#else
                DisplayBase::COL_SYS_PAL_443,
#endif
                FrameBuffer_Video,
                VIDEO_BUFFER_STRIDE,
#if VIDEO_INPUT_FORMAT == VIDEO_YCBCR422
                DisplayBase::VIDEO_FORMAT_YCBCR422,
                DisplayBase::WR_RD_WRSWA_NON,
#elif VIDEO_INPUT_FORMAT == VIDEO_RGB565
                DisplayBase::VIDEO_FORMAT_RGB565,
                DisplayBase::WR_RD_WRSWA_32_16BIT,
#else
                DisplayBase::VIDEO_FORMAT_RGB888,
                DisplayBase::WR_RD_WRSWA_32BIT,
#endif
                PIXEL_VW,
                PIXEL_HW
            );
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        while (1);
    }

    /* Interrupt callback function setting (Field end signal for recording function in scaler 0) */
    error = Display.Graphics_Irq_Handler_Set(VIDEO_INT_TYPE, 0, IntCallbackFunc_Vfield);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        while (1);
    }

    /* Video write process start */
    error = Display.Video_Start (VIDEO_INPUT_CH);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        while (1);
    }

    /* Video write process stop */
    error = Display.Video_Stop (VIDEO_INPUT_CH);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        while (1);
    }

    /* Video write process start */
    error = Display.Video_Start (VIDEO_INPUT_CH);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        while (1);
    }

    /* Wait vsync to update resister */
    WaitVsync(1);

#if(1) //lcd
    Display.Graphics_Lvds_Port_Init(lvds_pin, 8);
    rect.vs = 0;
    rect.vw = PIXEL_VW;
    rect.hs = 0;
    rect.hw = PIXEL_HW;

    Display.Graphics_Read_Setting(
        DisplayBase::GRAPHICS_LAYER_0,
        (void *)FrameBuffer_Video,
        VIDEO_BUFFER_STRIDE,
        DisplayBase::GRAPHICS_FORMAT_YCBCR422,
        DisplayBase::WR_RD_WRSWA_NON,
        &rect
    );
    Display.Graphics_Start(DisplayBase::GRAPHICS_LAYER_0);

    lcd_cntrst.write(1.0);

    while (1) {
        uint8_t count;
        uint32_t x0;
        uint32_t y0;
        uint32_t x1;
        uint32_t y1;

        count = 0;
        x0 = 0;
        y0 = 0;
        x1 = 0;
        y1 = 0;
        get_coordinates(&count, &x0, &y0, &x1, &y1);
        if (count != 0) {
            printf("count=%d, {X=%d,Y=%d}, {X=%d,Y=%d} \n", count, x0, y0, x1, y1);
        }
        Thread::wait(100);
    }
#endif
}
