/*
 * Copyright (c) 2015, 2016 ARM Limited. All rights reserved.
 * SPDX-License-Identifier: Apache-2.0
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#define __STDC_FORMAT_MACROS
#include <inttypes.h>
#include "simpleclient.h"
#include <string>
#include <sstream>
#include <vector>
#include "mbed-trace/mbed_trace.h"
#include "mbedtls/entropy_poll.h"

#include "security.h"

#include "mbed.h"
#include "DisplayBace.h"
#include "rtos.h"
#include "LCD_shield_config_4_3inch.h"
#include "recognition_proc.h"
#include "touch_proc.h"

// easy-connect compliancy, it has 2 sets of wifi pins we have only one
#define MBED_CONF_APP_ESP8266_TX MBED_CONF_APP_WIFI_TX
#define MBED_CONF_APP_ESP8266_RX MBED_CONF_APP_WIFI_RX
#include "easy-connect.h"


// These are example resource values for the Device Object
struct MbedClientDevice device = {
    "Manufacturer_String",      // Manufacturer
    "Type_String",              // Type
    "ModelNumber_String",       // ModelNumber
    "SerialNumber_String"       // SerialNumber
};

// Instantiate the class which implements LWM2M Client API (from simpleclient.h)
MbedClient mbed_client(device);

static DisplayBase Display;
static DigitalOut  lcd_pwon(P7_15);
static DigitalOut  lcd_blon(P8_1);
static PwmOut      lcd_cntrst(P8_15);
static Thread      recognitionTask;
static Thread      touchTask;
static char send_cloud_buf_last[1024 * 4];

// LED Output
DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);

/*
 * The Led contains one property (pattern) and a function (blink).
 * When the function blink is executed, the pattern is read, and the LED
 * will blink based on the pattern.
 */
class LedResource {
public:
    LedResource() {
        // create ObjectID with metadata tag of '3201', which is 'digital output'
        led_object = M2MInterfaceFactory::create_object("3201");
        M2MObjectInstance* led_inst = led_object->create_object_instance();

        // 5855 = Multi-state output
        M2MResource* color_res = led_inst->create_dynamic_resource("5855", "Color",
            M2MResourceInstance::STRING, false);
        // read and write
        color_res->set_operation(M2MBase::GET_PUT_ALLOWED);
        // set red as initial color
        color_res->set_value((const uint8_t*)"red", 3);
        
        // 5853 = Multi-state output
        M2MResource* pattern_res = led_inst->create_dynamic_resource("5853", "Pattern",
            M2MResourceInstance::STRING, false);
        // read and write
        pattern_res->set_operation(M2MBase::GET_PUT_ALLOWED);
        // set initial pattern (toggle every 200ms. 7 toggles in total)
        pattern_res->set_value((const uint8_t*)"500:500:500:500:500:500:500", 27);

        // there's not really an execute LWM2M ID that matches... hmm...
        M2MResource* led_res = led_inst->create_dynamic_resource("5850", "Blink",
            M2MResourceInstance::OPAQUE, false);
        // we allow executing a function here...
        led_res->set_operation(M2MBase::POST_ALLOWED);
        // when a POST comes in, we want to execute the led_execute_callback
        led_res->set_execute_function(execute_callback(this, &LedResource::blink));
    }

    M2MObject* get_object() {
        return led_object;
    }

    void blink(void *) {
        // read the value of 'Pattern'
        M2MObjectInstance* inst = led_object->object_instance();
        M2MResource* res = inst->resource("5853");
        // read the value of 'Color'
        M2MObjectInstance* instC = led_object->object_instance();
        M2MResource* resC = instC->resource("5855");

        // values in mbed Client are all buffers, and we need a vector of int's
        uint8_t* buffIn = NULL;
        uint32_t sizeIn;
        res->get_value(buffIn, sizeIn);
        
        uint8_t* cbuffIn = NULL;
        uint32_t csizeIn;
        resC->get_value(cbuffIn, csizeIn);

        // turn the buffer into a string, and initialize a vector<int> on the heap
        std::string s((char*)buffIn, sizeIn);
        std::vector<uint32_t>* v = new std::vector<uint32_t>;

        printf("led_execute_callback pattern=%s\r\n", s.c_str());

        // our pattern is something like 500:200:500, so parse that
        std::size_t found = s.find_first_of(":");
        while (found!=std::string::npos) {

            v->push_back(atoi((const char*)s.substr(0,found).c_str()));
            s = s.substr(found+1);
            found=s.find_first_of(":");
            if(found == std::string::npos) {
                v->push_back(atoi((const char*)s.c_str()));
            }
        }

        int position = 0;
        while (1) {
            do_blink(cbuffIn);
            if (position >= v->size()) {
                break;
            }
            // how long do we need to wait before the next blink?
            Thread::wait(v->at(position));
            position++;
        }
        free(buffIn);
        free(cbuffIn);
        delete v;
    }

private:
    M2MObject* led_object;

    void do_blink(uint8_t* color) {
        if (!strcmp((char *)color, "red")) {
            // blink the LED in red color
            led1 = !led1;
        }
        else if (!strcmp((char *)color, "green")) {
            // blink in green color
            led2 = !led2;
        }
        else if (!strcmp((char *)color, "blue")) {
            // blink in blue color
            led3 = !led3;
        }
        else if (!strcmp((char *)color, "cyan")) {
            // blink in cyan color
            led2 = !led2;
            led3 = !led3;
        }
        else if (!strcmp((char *)color, "yellow")) {
            // blink in yellow color
            led1 = !led1;
            led2 = !led2;
        }
        else if (!strcmp((char *)color, "magenta")) {
            // blink in magenta color
            led1 = !led1;
            led3 = !led3;
        }            
        else if (!strcmp((char *)color, "white")) {
            // blink in white color
            led1 = !led1;
            led2 = !led2;
            led3 = !led3;
        }
        else {
            // no operation
        }
    }
};

/*
 * The HVC contains a function (send string).
 * When `handle_string_send` is executed, the string after decoding is sent.
 */
class HVCResource {
public:
    HVCResource() {
        // create ObjectID with metadata tag of '3202', which is 'send string'
        hvc_object = M2MInterfaceFactory::create_object("3202");
        M2MObjectInstance* hvc_inst = hvc_object->create_object_instance();
        // create resource with ID '5700', which is 'send string'
        M2MResource* hvc_res = hvc_inst->create_dynamic_resource("5700", "hvc",
            M2MResourceInstance::STRING, true);
        // we can read this value
        hvc_res->set_operation(M2MBase::GET_ALLOWED);
        // set initial value (all values in mbed Client are buffers)
        // to be able to read this data easily in the Connector console, we'll use a string
        hvc_res->set_value((uint8_t*)"0", 1);
    }

    ~HVCResource() {
    }

    M2MObject* get_object() {
        return hvc_object;
    }

    /*
     * When you success the decode process of barcode, we send the string after decoding to mbed Device Connector.
     */
    void handle_string_send(char * addr, int size) {
        M2MObjectInstance* inst = hvc_object->object_instance();
        M2MResource* res = inst->resource("5700");

        printf("%s\r\n", addr);

        // tell the string to connector
        res->set_value((uint8_t *)addr, size);
    }

private:
    M2MObject* hvc_object;
};

// Network interaction must be performed outside of interrupt context
Semaphore updates(0);
osThreadId mainThread;

HVCResource hvc_resource;
volatile bool sent = false;
Timer send_timer;

void HVCSendData(char * addr, int size) {
    if (send_timer.read_ms() > 500) {
        if (memcmp(send_cloud_buf_last, addr, sizeof(send_cloud_buf_last)) != 0) {
            memcpy(send_cloud_buf_last, addr, sizeof(send_cloud_buf_last));
            send_timer.reset();
            hvc_resource.handle_string_send(addr, size);
            sent = true;
            updates.release();
        }
    }
}

// LCD
static void IntCallbackFunc_LoVsync(DisplayBase::int_type_t int_type) {
    /* Interrupt callback function for Vsync interruption */
    touch_lcd_int(int_type);
}

static void Init_LCD_Display(void) {
    DisplayBase::graphics_error_t error;
    DisplayBase::lcd_config_t lcd_config;
    PinName lvds_pin[8] = {
        /* data pin */
        P5_7, P5_6, P5_5, P5_4, P5_3, P5_2, P5_1, P5_0
    };

    lcd_pwon = 0;
    lcd_blon = 0;
    Thread::wait(100);
    lcd_pwon = 1;
    lcd_blon = 1;

    Display.Graphics_Lvds_Port_Init(lvds_pin, 8);

    /* Graphics initialization process */
    lcd_config = LcdCfgTbl_LCD_shield;
    error = Display.Graphics_init(&lcd_config);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        mbed_die();
    }

    /* Interrupt callback function setting (Vsync signal output from scaler 0) */
    error = Display.Graphics_Irq_Handler_Set(DisplayBase::INT_TYPE_S0_LO_VSYNC, 0, IntCallbackFunc_LoVsync);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        mbed_die();
    }
}

// Entry point to the program
int main() {

    unsigned int seed;
    size_t len;

    /* Initialization of LCD */
    Init_LCD_Display();

    /* Start recognition processing */
    recognitionTask.start(callback(recognition_task, &Display));

    /* Start touch panel processing */
    touchTask.start(callback(touch_task, &Display));

    /* Backlight on */
    Thread::wait(200);
    lcd_cntrst.write(1.0);

#ifdef MBEDTLS_ENTROPY_HARDWARE_ALT
    // Used to randomize source port
    mbedtls_hardware_poll(NULL, (unsigned char *) &seed, sizeof seed, &len);

#elif defined MBEDTLS_TEST_NULL_ENTROPY

#warning "mbedTLS security feature is disabled. Connection will not be secure !! Implement proper hardware entropy for your selected hardware."
    // Used to randomize source port
    mbedtls_null_entropy_poll( NULL,(unsigned char *) &seed, sizeof seed, &len);

#else

#error "This hardware does not have entropy, endpoint will not register to Connector.\
You need to enable NULL ENTROPY for your application, but if this configuration change is made then no security is offered by mbed TLS.\
Add MBEDTLS_NO_DEFAULT_ENTROPY_SOURCES and MBEDTLS_TEST_NULL_ENTROPY in mbed_app.json macros to register your endpoint."

#endif

    srand(seed);
    // Keep track of the main thread
    mainThread = osThreadGetId();
    printf("\nStarting mbed Client example in ");
#if defined (MESH) || (MBED_CONF_LWIP_IPV6_ENABLED==true)
    printf("IPv6 mode\n");
#else
    printf("IPv4 mode\n");
#endif

    mbed_trace_init();

    NetworkInterface* network = easy_connect(true);
    if(network == NULL) {
        printf("\nConnection to Network Failed - exiting application...\n");
        return -1;
    }

    // we create our button and LED resources
    LedResource led_resource;


    // Create endpoint interface to manage register and unregister
    mbed_client.create_interface(MBED_SERVER_ADDRESS, network);

    // Create Objects of varying types, see simpleclient.h for more details on implementation.
    M2MSecurity* register_object = mbed_client.create_register_object(); // server object specifying connector info
    M2MDevice*   device_object   = mbed_client.create_device_object();   // device resources object

    // Create list of Objects to register
    M2MObjectList object_list;

    // Add objects to list
    object_list.push_back(device_object);
    object_list.push_back(led_resource.get_object());
    object_list.push_back(hvc_resource.get_object());

    // Set endpoint registration object
    mbed_client.set_register_object(register_object);

    // Register with mbed Device Connector
    mbed_client.test_register(register_object, object_list);
    send_timer.start();
    while (1) {
        updates.wait(25000);
        if(!sent) {
            mbed_client.test_update_register();
        }
        sent = false;
    }
}
