#include "mbed.h"
#include "DisplayBace.h"
#include "rtos.h"
#include "RGA.h"
#include "BinaryImage_RZ_A1H.h"
#include "FATFileSystem.h"
#include "SDBlockDevice_GR_PEACH.h"
#include "USBHostMSD.h"
#include "USBHostMouse.h"
#include "scan_folder.h"

/**** User Selection *********/
#define WAIT_TIME                           (10000) /* wait time (ms) 0:infinite */
#define DISSOLVE_STEP_NUM                   (16)    /* minimum 1 */
#define SCROLL_STEP_NUM                     (8)     /* minimum 1 */
#define SCROLL_DIRECTION                    (-1)    /* Select 1(left to right) or -1(right to left) */
/** LCD setting **/
#define LCD_TYPE                            (0)     /* Select 0(4.3inch) , 1(7.1inch), 2(Display shield) or 3(RSK TFT) */
/*****************************/

#define GRAPHICS_FORMAT                     (DisplayBase::GRAPHICS_FORMAT_RGB565)
#define WR_RD_WRSWA                         (DisplayBase::WR_RD_WRSWA_32_16BIT)

/** LCD shield config **/
#if (LCD_TYPE == 0)
  #include "LCD_shield_config_4_3inch.h"
#elif (LCD_TYPE == 1)
  #include "LCD_shield_config_7_1inch.h"
#elif (LCD_TYPE == 2)
  #include "Display_shield_config.h"
#else
  #include "LCD_shield_config_RSK_TFT.h"
#endif

#if (SCROLL_DIRECTION == -1)
#define SCROLL_DIRECTION_NEXT               (-1)
#define SCROLL_DIRECTION_PREV               (1)
#else
#define SCROLL_DIRECTION_NEXT               (1)
#define SCROLL_DIRECTION_PREV               (-1)
#endif

/* FRAME BUFFER Parameter */
#define FRAME_BUFFER_BYTE_PER_PIXEL         (2)
#define FRAME_BUFFER_STRIDE                 (((LCD_PIXEL_WIDTH * FRAME_BUFFER_BYTE_PER_PIXEL) + 31u) & ~31u)

#define EVENT_NONE                          (0)
#define EVENT_DISP                          (1)
#define EVENT_NEXT                          (2)
#define EVENT_PREV                          (3)
#define EVENT_MOVE                          (4)
#define EVENT_MOVE_END                      (5)

#define MAX_JPEG_SIZE                       (1024 * 450)
#define MOUNT_NAME                          "storage"

#define TOUCH_NUM                           (2u)

typedef struct {
    int  pos_x;
    int  pos_y;
    bool disp_pos;
    bool disp_time;
    bool change;
} mouse_info_t;

typedef struct {
    int       event_req;
    uint8_t * p_pic_next;
    uint8_t * p_pic_now;
    float32_t magnification;
    int       x_move;
    int       swipe_end_move;
    int       drow_pos_x;
    int       drow_pos_y;
    int       scale_end_move_x;
    int       scale_end_move_y;
    int       min_x;
    int       min_y;
    bool      scroll;
} efect_info_t;

static const char_t * extension_tbl[] = {
    ".jpg",
    NULL
};
static InterruptIn button(USER_BUTTON0);
static DisplayBase Display;
static Canvas2D_ContextClass canvas2d;
#if (LCD_TYPE != 2)
static DigitalOut  lcd_pwon(P7_15);
static DigitalOut  lcd_blon(P8_1);
static PwmOut      lcd_cntrst(P8_15);
#endif
static Serial pc(USBTX, USBRX);
static Timer system_timer;
static Semaphore   sem_touch_int(0);
static Thread mouseTask(osPriorityNormal, 1024);
#ifdef TouckKey_LCD_shield
static Thread touchTask(osPriorityNormal, 1024 * 8);
#endif

#if defined(__ICCARM__)
#pragma data_alignment=32
static uint8_t user_frame_buffer1[FRAME_BUFFER_STRIDE * LCD_PIXEL_HEIGHT];
#pragma data_alignment=32
static uint8_t user_frame_buffer2[FRAME_BUFFER_STRIDE * LCD_PIXEL_HEIGHT];
#pragma data_alignment=8
static uint8_t JpegBuffer[MAX_JPEG_SIZE]@ ".mirrorram";  //8 bytes aligned!;
#else
static uint8_t user_frame_buffer1[FRAME_BUFFER_STRIDE * LCD_PIXEL_HEIGHT]__attribute((aligned(32)));  /* 32 bytes aligned */
static uint8_t user_frame_buffer2[FRAME_BUFFER_STRIDE * LCD_PIXEL_HEIGHT]__attribute((aligned(32))); /* 32 bytes aligned */
static uint8_t JpegBuffer[MAX_JPEG_SIZE]__attribute((section("NC_BSS"),aligned(8)));  //8 bytes aligned!;
#endif
static frame_buffer_t frame_buffer_info;
static volatile int32_t vsync_count = 0;
static ScanFolder scan_foler;

static mouse_info_t mouse_info = {0, 0, false, false, false};

static int disp_wait_time = WAIT_TIME;
static uint32_t file_id_now = 0xffffffff;
static uint32_t file_id_next = 0xffffffff;
static int dissolve_seq = 0;
static efect_info_t efect_info;

#define ROM_PIC_NUM             (4)
static bool storage_mode = false;
static uint32_t total_file_num = ROM_PIC_NUM;

static const graphics_image_t* number_tbl[10] = {
    number0_Img,
    number1_Img,
    number2_Img,
    number3_Img,
    number4_Img,
    number5_Img,
    number6_Img,
    number7_Img,
    number8_Img,
    number9_Img
};

/****** LCD ******/
static void IntCallbackFunc_LoVsync(DisplayBase::int_type_t int_type) {
    /* Interrupt callback function for Vsync interruption */
    if (vsync_count > 0) {
        vsync_count--;
    }
}

static void Wait_Vsync(const int32_t wait_count) {
    /* Wait for the specified number of times Vsync occurs */
    vsync_count = wait_count;
    while (vsync_count > 0) {
        /* Do nothing */
    }
}

static void Init_LCD_Display(void) {
    DisplayBase::graphics_error_t error;
    DisplayBase::lcd_config_t lcd_config;
#if (LCD_TYPE == 2)
    PinName lcd_pin[28] = {
        /* data pin */
        P11_15, P11_14, P11_13, P11_12, P5_7, P5_6, P5_5, P5_4, P5_3, P5_2, P5_1, P5_0,
        P4_7, P4_6, P4_5, P4_4, P10_12, P10_13, P10_14, P10_15, P3_15, P3_14, P3_13,
        P3_12, P3_11, P3_10, P3_9, P3_8
    };
    Display.Graphics_Lcd_Port_Init(lcd_pin, 28);
#else
    PinName lvds_pin[8] = {
        /* data pin */
        P5_7, P5_6, P5_5, P5_4, P5_3, P5_2, P5_1, P5_0
    };

    lcd_pwon = 0;
    lcd_blon = 0;
    Thread::wait(100);
    lcd_pwon = 1;
    lcd_blon = 1;

    Display.Graphics_Lvds_Port_Init(lvds_pin, 8);
#endif

    /* Graphics initialization process */
    lcd_config = LcdCfgTbl_LCD_shield;
    error = Display.Graphics_init(&lcd_config);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        mbed_die();
    }

    /* Interrupt callback function setting (Vsync signal output from scaler 0) */
    error = Display.Graphics_Irq_Handler_Set(DisplayBase::INT_TYPE_S0_LO_VSYNC, 0, IntCallbackFunc_LoVsync);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        mbed_die();
    }
}

static void Start_LCD_Display(uint8_t * p_buf) {
    DisplayBase::rect_t rect;

    rect.vs = 0;
    rect.vw = LCD_PIXEL_HEIGHT;
    rect.hs = 0;
    rect.hw = LCD_PIXEL_WIDTH;
    Display.Graphics_Read_Setting(
        DisplayBase::GRAPHICS_LAYER_0,
        (void *)p_buf,
        FRAME_BUFFER_STRIDE,
        GRAPHICS_FORMAT,
        WR_RD_WRSWA,
        &rect
    );
    Display.Graphics_Start(DisplayBase::GRAPHICS_LAYER_0);
}

static void Update_LCD_Display(frame_buffer_t * frmbuf_info) {
    Display.Graphics_Read_Change(DisplayBase::GRAPHICS_LAYER_0,
     (void *)frmbuf_info->buffer_address[frmbuf_info->draw_buffer_index]);
    Wait_Vsync(1);
}

static void Swap_FrameBuffer(frame_buffer_t * frmbuf_info) {
    if (frmbuf_info->draw_buffer_index == 1) {
        frmbuf_info->draw_buffer_index = 0;
    } else {
        frmbuf_info->draw_buffer_index = 1;
    }
}


/****** Draw Image ******/
static void draw_mouse_pos(void) {
    if (mouse_info.disp_pos != false) {
        /* Draw a image */
        canvas2d.drawImage(mousu_pos_Img, mouse_info.pos_x, mouse_info.pos_y);
        R_OSPL_CLEAR_ERROR();
    }
}

static void draw_disp_time(void) {
    if (mouse_info.disp_time != false) {
        int wk_ofs = (LCD_PIXEL_WIDTH - 215) / 2 + 103;
        int wk_sec;

        /* Draw a image */
        canvas2d.drawImage(disp_xsec_Img, (LCD_PIXEL_WIDTH - 215) / 2, LCD_PIXEL_HEIGHT / 2);
        R_OSPL_CLEAR_ERROR();

        if (disp_wait_time == 0) {
            /* Draw a image */
            canvas2d.drawImage(number_inf_Img, wk_ofs, LCD_PIXEL_HEIGHT / 2);
            R_OSPL_CLEAR_ERROR();
        } else {
            wk_sec = (disp_wait_time / 10000) % 10;
            if (wk_sec != 0) {
                /* Draw a image */
                canvas2d.drawImage(number_tbl[wk_sec], wk_ofs, LCD_PIXEL_HEIGHT / 2);
                R_OSPL_CLEAR_ERROR();
                wk_ofs += 12;
            } else {
                wk_ofs += 6;
            }
            wk_sec = (disp_wait_time / 1000) % 10;
            /* Draw a image */
            canvas2d.drawImage(number_tbl[wk_sec], wk_ofs, LCD_PIXEL_HEIGHT / 2);
            R_OSPL_CLEAR_ERROR();
        }
    }
}

static void draw_image_scroll(frame_buffer_t* frmbuf_info, const graphics_image_t* image_last,
                              const graphics_image_t* image_new, float32_t scroll, int ditection) {
    Swap_FrameBuffer(frmbuf_info);
    /* Clear */
    canvas2d.clearRect(0, 0, frmbuf_info->width, frmbuf_info->height);
    /* Draw a image */
    canvas2d.globalAlpha = 1.0f;
    canvas2d.drawImage((const graphics_image_t*)image_last,
                       (int_t)(frmbuf_info->width * scroll) * ditection, 0,
                       frmbuf_info->width, frmbuf_info->height);
    R_OSPL_CLEAR_ERROR();
    canvas2d.globalAlpha = 1.0f;
    canvas2d.drawImage((const graphics_image_t*)image_new,
                       ((int_t)(frmbuf_info->width * scroll) - frmbuf_info->width) * ditection, 0,
                       frmbuf_info->width, frmbuf_info->height);
    R_OSPL_CLEAR_ERROR();
    /* mouse pos */
    draw_mouse_pos();
    draw_disp_time();
    /* Complete drawing */
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
    Update_LCD_Display(frmbuf_info);
}

static void draw_image_dissolve(frame_buffer_t* frmbuf_info, const graphics_image_t* image_last,
                                const graphics_image_t* image_new, float32_t alpha) {
    Swap_FrameBuffer(frmbuf_info);
    /* Clear */
    canvas2d.clearRect(0, 0, frmbuf_info->width, frmbuf_info->height);
    /* Draw a image */
    canvas2d.globalAlpha = 1.0f - alpha;
    canvas2d.drawImage((const graphics_image_t*)image_last, 0, 0, frmbuf_info->width, frmbuf_info->height);
    R_OSPL_CLEAR_ERROR();
    canvas2d.globalAlpha = alpha;
    canvas2d.drawImage((const graphics_image_t*)image_new, 0, 0, frmbuf_info->width, frmbuf_info->height);
    R_OSPL_CLEAR_ERROR();
    /* mouse pos */
    draw_mouse_pos();
    draw_disp_time();
    /* Complete drawing */
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
    Update_LCD_Display(frmbuf_info);
}

static void draw_image(frame_buffer_t* frmbuf_info, const graphics_image_t* image_new, 
 uint32_t pos_x, uint32_t pos_y, graphics_matrix_float_t zoom) {
    int_t dest_width;
    int_t dest_height;

    Swap_FrameBuffer(frmbuf_info);
    /* Clear */
    canvas2d.clearRect(0, 0, frmbuf_info->width, frmbuf_info->height);
    /* Draw a image */
    canvas2d.globalAlpha = 1.0f;
    dest_width  = frmbuf_info->width * zoom;
    dest_height = frmbuf_info->height * zoom;
    canvas2d.drawImage((const graphics_image_t*)image_new,
                        pos_x, pos_y, dest_width, dest_height);
    R_OSPL_CLEAR_ERROR();
    /* mouse pos */
    draw_mouse_pos();
    draw_disp_time();
    /* Complete drawing */
    R_GRAPHICS_Finish(canvas2d.c_LanguageContext);
    Update_LCD_Display(frmbuf_info);
}


/****** File Access ******/
static bool read_new_file(uint8_t ** pp_buf, uint32_t file_id) {
    bool ret = false;

    if (storage_mode == false) {
        ret = true;
        switch (file_id) {
            case 0:     *pp_buf = (uint8_t *)img01_File;    break;
            case 1:     *pp_buf = (uint8_t *)img02_File;    break;
            case 2:     *pp_buf = (uint8_t *)img03_File;    break;
            case 3:     *pp_buf = (uint8_t *)img04_File;    break;
            default:    ret = false;                        break;
        }
    } else {
        FILE * fp = scan_foler.open(file_id);
        size_t read_size;

        read_size = fread(*pp_buf, sizeof(char), MAX_JPEG_SIZE, fp);
        scan_foler.close(fp);
        if (read_size < MAX_JPEG_SIZE) {
            ret = true;
        }
    }

    return ret;
}

static void swap_file_buff(void) {
    uint8_t * p_wk_pic = efect_info.p_pic_now;
    uint32_t wk_file_id = file_id_now;

    efect_info.p_pic_now = efect_info.p_pic_next;
    efect_info.p_pic_next = p_wk_pic;
    file_id_now = file_id_next;
    file_id_next = wk_file_id;
}

static bool read_next_file(void) {
    uint32_t wk_file_id;
    bool ret = false;

    if (file_id_now < (total_file_num - 1)) {
        wk_file_id = file_id_now + 1;
    } else {
        wk_file_id = 0;
    }
    if (wk_file_id != file_id_next) {
        ret = read_new_file(&efect_info.p_pic_next, wk_file_id);
        if (ret != false) {
            file_id_next = wk_file_id;
        }
    } else {
        ret = true;
    }

    return ret;
}

static bool read_prev_file(void) {
    uint32_t wk_file_id;
    bool ret = false;

    if (file_id_now >= total_file_num) {
        wk_file_id = 0;
    } else if (file_id_now != 0) {
        wk_file_id = file_id_now - 1;
    } else {
        wk_file_id = total_file_num - 1;
    }
    if (wk_file_id != file_id_next) {
        ret = read_new_file(&efect_info.p_pic_next, wk_file_id);
        if (ret != false) {
            file_id_next = wk_file_id;
        }
    } else {
        ret = true;
    }

    return ret;
}


/****** Efect setting ******/
static void zoom_scroll(int x, int y) {
    efect_info.drow_pos_x += x;
    if (efect_info.drow_pos_x < efect_info.min_x) {
        efect_info.drow_pos_x = efect_info.min_x;
    }
    if (efect_info.drow_pos_x > 0) {
        efect_info.drow_pos_x = 0;
    }
    efect_info.drow_pos_y += y;
    if (efect_info.drow_pos_y < efect_info.min_y) {
        efect_info.drow_pos_y = efect_info.min_y;
    }
    if (efect_info.drow_pos_y > 0) {
        efect_info.drow_pos_y = 0;
    }
}

static void move_func(int x, int y) {
    if (efect_info.magnification != 1.0f) {
        zoom_scroll(x, y);
        efect_info.event_req = EVENT_DISP;
    } else {
        efect_info.scroll = true;
        efect_info.x_move += x;
        efect_info.event_req = EVENT_MOVE;
    }
}

static void move_end_func(int x, int y) {
    if (efect_info.magnification != 1.0f) {
        efect_info.scale_end_move_x = x;
        efect_info.scale_end_move_y = y;
    } else {
        if (efect_info.scroll != false) {
            efect_info.scroll = false;
            if (efect_info.x_move != 0) {
                efect_info.swipe_end_move = x;
                efect_info.event_req = EVENT_MOVE_END;
            }
        }
    }
}

static void zoom_func(uint32_t center_x_last, uint32_t center_y_last, uint32_t center_x, uint32_t center_y, float32_t new_magnification) {
    if (new_magnification != efect_info.magnification) {
        if (new_magnification < 1.0f) {
            new_magnification = 1.0f;
        }
        if (new_magnification > 10.0f) {
            new_magnification = 10.0f;
        }
        efect_info.min_x =  0 - (frame_buffer_info.width * (new_magnification - 1.0f));
        efect_info.min_y =  0 - (frame_buffer_info.height * (new_magnification - 1.0f));

        efect_info.drow_pos_x = -((center_x_last - efect_info.drow_pos_x) / efect_info.magnification * new_magnification - center_x);
        if (efect_info.drow_pos_x < efect_info.min_x) {
            efect_info.drow_pos_x = efect_info.min_x;
        }
        if (efect_info.drow_pos_x > 0) {
            efect_info.drow_pos_x = 0;
        }

        efect_info.drow_pos_y = -((center_y_last - efect_info.drow_pos_y) / efect_info.magnification * new_magnification - center_y);
        if (efect_info.drow_pos_y < efect_info.min_y) {
            efect_info.drow_pos_y = efect_info.min_y;
        }
        if (efect_info.drow_pos_y > 0) {
            efect_info.drow_pos_y = 0;
        }
        if (new_magnification == 1.0f) {
            efect_info.drow_pos_x = 0;
            efect_info.drow_pos_y = 0;
        }
        efect_info.x_move = 0;
        efect_info.magnification = new_magnification;
        if (efect_info.event_req == EVENT_NONE) {
            efect_info.event_req = EVENT_DISP;
        }
    }
}

/****** Mouse ******/
void onMouseEvent(uint8_t buttons, int8_t x, int8_t y, int8_t z) {
    static uint8_t last_buttons = 0;
    int wk_pos;
    int wk_disp_time;
    int mouse_move_x;
    int mouse_move_y;

//    printf("buttons: %d, x: %d, y: %d, z: %d\r\n", buttons, x, y, z);

    wk_pos = mouse_info.pos_x;
    wk_pos += x;
    mouse_move_x = x;
    if (wk_pos < 0) {
        wk_pos = 0;
        mouse_move_x = 0;
    }
    if (wk_pos > (LCD_PIXEL_WIDTH - 10)) {
        wk_pos = LCD_PIXEL_WIDTH - 10;
        mouse_move_x = 0;
    }
    mouse_info.pos_x = wk_pos;

    wk_pos = mouse_info.pos_y;
    wk_pos += y;
    mouse_move_y = y;
    if (wk_pos < 0) {
        wk_pos = 0;
        mouse_move_y = 0;
    }
    if (wk_pos > (LCD_PIXEL_HEIGHT - 10)) {
        wk_pos = (LCD_PIXEL_HEIGHT - 10);
        mouse_move_y = 0;
    }
    mouse_info.pos_y = wk_pos;

    if (mouse_info.disp_pos != false) {
        if ((buttons & 0x01) != 0) {
            move_func(mouse_move_x, mouse_move_y);
        } else if (((buttons & 0x01) == 0) && ((last_buttons & 0x01) != 0)) {
            move_end_func(mouse_move_x, mouse_move_y);
        } else if ((efect_info.event_req == EVENT_NONE) && (dissolve_seq == 0)) {
            efect_info.event_req = EVENT_DISP;
        } else {
            /* do nothing */
        }
    } else {
        // left
        if (((buttons & 0x01) == 0) && ((last_buttons & 0x01) != 0)) {
            efect_info.event_req = EVENT_NEXT;
        }

        // rigth
        if (((buttons & 0x02) == 0) && ((last_buttons & 0x02) != 0)) {
            efect_info.event_req = EVENT_PREV;
        }
    }

    if (((buttons & 0x04) != 0) && ((last_buttons & 0x04) == 0)) {
        mouse_info.disp_pos = !mouse_info.disp_pos;
        if ((efect_info.event_req == EVENT_NONE) && (dissolve_seq == 0)) {
            efect_info.event_req = EVENT_DISP;
        }
    }

    if (z != 0) {
        if (mouse_info.disp_pos == false) {
            wk_disp_time = disp_wait_time;
            if (z > 0) {
                wk_disp_time += 1000;
                if (wk_disp_time > 15000) {
                    wk_disp_time = 15000;
                }
            } else {
                wk_disp_time -= 1000;
                if (wk_disp_time < 1000) {
                    wk_disp_time = 0;
                }
            }
            disp_wait_time = wk_disp_time;
            system_timer.reset();
            system_timer.start();
            mouse_info.disp_time = true;
            if ((efect_info.event_req == EVENT_NONE) && (dissolve_seq == 0)) {
                efect_info.event_req = EVENT_DISP;
            }
        } else {
            zoom_func(mouse_info.pos_x + 3, mouse_info.pos_y + 3,
                      mouse_info.pos_x + 3, mouse_info.pos_y + 3,
                      efect_info.magnification * (1.0f + (float_t)z / 10.0f));
        }
    }

    last_buttons = buttons;
}

void mouse_task(void) {
    USBHostMouse mouse;

    while (1) {
        /* try to connect a USB mouse */
        while (!mouse.connect()) {
            Thread::wait(500);
        }

        /* when connected, attach handler called on mouse event */
        mouse.attachEvent(onMouseEvent);

        /* wait until the mouse is disconnected */
        while (mouse.connected()) {
            Thread::wait(500);
        }
        mouse_info.disp_pos = false;
        mouse_info.disp_time = false;
        if ((efect_info.event_req == EVENT_NONE) && (dissolve_seq == 0)) {
            efect_info.event_req = EVENT_DISP;
        }
    }
}


/****** Touch ******/
#ifdef TouckKey_LCD_shield
static float32_t get_distance(uint32_t x0, uint32_t y0, uint32_t x1, uint32_t y1) {
    float32_t distance_x;
    float32_t distance_y;

    if (x0 > x1) {
        distance_x = x0 - x1;
    } else {
        distance_x = x1 - x0;
    }
    if (y0 > y1) {
        distance_y = y0 - y1;
    } else {
        distance_y = y1 - y0;
    }
    return hypotf(distance_x, distance_y);
}

static uint32_t get_center(uint32_t xy0, uint32_t xy1) {
    uint32_t center_pos;

    if (xy0 > xy1) {
        center_pos = (xy0 - xy1) / 2 + xy1;
    } else {
        center_pos = (xy1 - xy0) / 2 + xy0;
    }

    return center_pos;
}

static void touch_int_callback(void) {
    sem_touch_int.release();
}

static void touch_task(void) {
    bool skip = false;
    bool zoom_on = false;
    uint32_t center_x;
    uint32_t center_y;
    uint32_t center_x_last = 0;
    uint32_t center_y_last = 0;
    int click_cnt = 0;
    int move_x = 0;
    int move_y = 0;
    float32_t distance;
    float32_t distance_last;
    int down_time = 0;;
    int last_down_time = 0;
    int event_time = 0;;
    int event_time_last = 0;
    TouchKey::touch_pos_t touch_pos[TOUCH_NUM];
    TouchKey::touch_pos_t touch_pos_last[TOUCH_NUM];
    int touch_num = 0;
    int touch_num_last = 0;
    Timer touch_timer;
    TouckKey_LCD_shield touch(P4_0, P2_13, I2C_SDA, I2C_SCL);

    /* Callback setting */
    touch.SetCallback(&touch_int_callback);

    /* Reset touch IC */
    touch.Reset();

    touch_timer.reset();
    touch_timer.start();

    while (1) {
        /* Wait touch event */
        sem_touch_int.wait();

        /* Get touch coordinates */
        touch_num = touch.GetCoordinates(TOUCH_NUM, touch_pos);
        event_time = touch_timer.read_ms();
        if (touch_num > touch_num_last) {
            if (touch_num_last == 0) {
                down_time = event_time;
            }
            if (touch_num == 2) {
                zoom_on = true;
            }
            move_x = 0;
            move_y = 0;
            distance_last = 0;
            if ((down_time - last_down_time) > 500) {
                click_cnt = 0;
            }
            last_down_time = down_time;
        } else if ((touch_num == 0) && (touch_num_last != 0)) {
            if (((event_time - down_time) < 200)
             && (abs(move_x) < 10) && (abs(move_y) < 10)) {
                click_cnt++;
                if (click_cnt == 2) {
                    zoom_func(0, 0, 0, 0, 1.0f);
                }
                move_x = 0;
                move_y = 0;
            } else {
                click_cnt = 0;
            }
            if (zoom_on == false) {
                move_end_func(move_x, move_y);
            }
            zoom_on = false;
            distance_last = 0;
        } else if ((touch_num == 1) && (touch_num_last == 2)) {
            distance_last = 0;
        } else if ((touch_num != 0) && ((event_time - event_time_last) >= 50)) {
            event_time_last = event_time;
            if (touch_num == 1) {
                if (zoom_on == false) {
                    move_x = (touch_pos[0].x - touch_pos_last[0].x);
                    move_y = (touch_pos[0].y - touch_pos_last[0].y);
                    move_func(move_x, move_y);
                }
            } else {
                center_x = get_center(touch_pos[0].x, touch_pos[1].x);
                center_y = get_center(touch_pos[0].y, touch_pos[1].y);
                distance = get_distance(touch_pos[0].x, touch_pos[0].y, touch_pos[1].x, touch_pos[1].y);
                if (distance < 1) {
                    distance = 1;
                }
                if (distance_last != 0) {
                    zoom_func(center_x_last, center_y_last, center_x, center_y, efect_info.magnification * (distance / distance_last));
                }
                center_x_last = center_x;
                center_y_last = center_y;
                distance_last = distance;
            }
        } else {
            skip = true;
        }
        if (skip == false) {
            touch_pos_last[0] = touch_pos[0];
            touch_pos_last[1] = touch_pos[1];
            touch_num_last = touch_num;
        }
        skip = false;
    }
}
#endif

static void button_rise(void) {
    int wk_disp_time;

    wk_disp_time = disp_wait_time;
    wk_disp_time += 1000;
    if (wk_disp_time > 15000) {
        wk_disp_time = 0;
    }
    disp_wait_time = wk_disp_time;
    system_timer.reset();
    system_timer.start();
    mouse_info.disp_time = true;
    if ((efect_info.event_req == EVENT_NONE) && (dissolve_seq == 0)) {
        efect_info.event_req = EVENT_DISP;
    }
}

int main(void) {
    errnum_t err;
    Canvas2D_ContextConfigClass config;

    /* Change the baud rate of the printf() */
    pc.baud(921600);

    /* Initialization of LCD */
    Init_LCD_Display();    /* When using LCD, please call before than Init_Video(). */

    memset(user_frame_buffer1, 0, sizeof(user_frame_buffer1));
    memset(user_frame_buffer2, 0, sizeof(user_frame_buffer2));
    frame_buffer_info.buffer_address[0] = user_frame_buffer1;
    frame_buffer_info.buffer_address[1] = user_frame_buffer2;
    frame_buffer_info.buffer_count      = 2;
    frame_buffer_info.show_buffer_index = 0;
    frame_buffer_info.draw_buffer_index = 0;
    frame_buffer_info.width             = LCD_PIXEL_WIDTH;
    frame_buffer_info.byte_per_pixel    = FRAME_BUFFER_BYTE_PER_PIXEL;
    frame_buffer_info.stride            = LCD_PIXEL_WIDTH * FRAME_BUFFER_BYTE_PER_PIXEL;
    frame_buffer_info.height            = LCD_PIXEL_HEIGHT;
    frame_buffer_info.pixel_format      = PIXEL_FORMAT_RGB565;

    config.frame_buffer = &frame_buffer_info;
    canvas2d = R_RGA_New_Canvas2D_ContextClass(config);
    err = R_OSPL_GetErrNum();
    if (err != 0) {
        printf("Line %d, error %d\n", __LINE__, err);
        mbed_die();
    }

    /* Start of LCD */
    Start_LCD_Display(frame_buffer_info.buffer_address[0]);

#if (LCD_TYPE != 2)
    /* Backlight on */
    Thread::wait(200);
    lcd_cntrst.write(1.0);
#endif

    mouseTask.start(callback(mouse_task));
#ifdef TouckKey_LCD_shield
    touchTask.start(callback(touch_task));
#endif
    button.rise(&button_rise);

    FATFileSystem fs(MOUNT_NAME);
    SDBlockDevice_GR_PEACH sd;
    USBHostMSD usb;
    int storage_type = 0;
    int next_storage_type = 0;
    int wait_time = 0;
    uint8_t * temp_buf[2];
    bool touch_key_in = false;
    int direction;
    int i;
    int wk_event_req;
    int type;
    int wk_abs_x_pos;

    temp_buf[0] = JpegBuffer;
    temp_buf[1] = new uint8_t[MAX_JPEG_SIZE];
    efect_info.event_req = EVENT_DISP;
    efect_info.p_pic_now = (uint8_t *)Renesas_logo_File;
    efect_info.p_pic_next = NULL;
    efect_info.magnification = 1.0f;
    efect_info.swipe_end_move = 0;
    efect_info.drow_pos_x = 0;
    efect_info.drow_pos_y = 0;
    efect_info.scale_end_move_x = 0;
    efect_info.scale_end_move_y = 0;
    efect_info.min_x = 0;
    efect_info.min_y = 0;
    efect_info.scroll = false;
    zoom_func(0, 0, 0, 0, 1.0f);

    while (1) {
        if (storage_type == 1) {
            if (!sd.connected()) {
                fs.unmount();
                next_storage_type = 0;
            }
        }
        if (storage_type == 2) {
            if (!usb.connected()) {
                fs.unmount();
                next_storage_type = 0;
            }
        }
        if (storage_type == 0) {
            if (sd.connect()) {
                next_storage_type = 1; // SD
                fs.mount(&sd);
            } else if (usb.connect()) {
                next_storage_type = 2; // USB
                fs.mount(&usb);
            } else {
                // do nothing
            }
        }
        if (next_storage_type != storage_type) {
            if (next_storage_type == 0) {
                storage_mode = false;
                total_file_num = ROM_PIC_NUM;
                file_id_now  = 0xffffffff;
                file_id_next = 0xffffffff;
                efect_info.p_pic_now    = (uint8_t *)Renesas_logo_File;
                efect_info.p_pic_next   = NULL;
                zoom_func(0, 0, 0, 0, 1.0f);
                efect_info.event_req    = EVENT_DISP;
            } else {
                storage_mode = true;
                scan_foler.scan("/"MOUNT_NAME, extension_tbl);
                total_file_num = scan_foler.getTotalFile();
                file_id_now  = 0;
                file_id_next = 0xffffffff;
                efect_info.p_pic_now    = temp_buf[0];
                efect_info.p_pic_next   = temp_buf[1];
                read_new_file(&efect_info.p_pic_now, file_id_now);
                zoom_func(0, 0, 0, 0, 1.0f);
                efect_info.event_req    = EVENT_DISP;
            }
            storage_type = next_storage_type;
        }

        wk_event_req = efect_info.event_req;
        efect_info.event_req = EVENT_NONE;

        if (wk_event_req != EVENT_NONE) {
            dissolve_seq = 0;
        }

        switch (wk_event_req) {
            case EVENT_NEXT:
            case EVENT_PREV:
                zoom_func(0, 0, 0, 0, 1.0f);
                if (wk_event_req == EVENT_NEXT) {
                    direction = SCROLL_DIRECTION_NEXT;
                    read_next_file();
                } else {
                    direction = SCROLL_DIRECTION_PREV;
                    read_prev_file();
                }
                for (i = 1; i <= SCROLL_STEP_NUM; i++) {
                    draw_image_scroll(&frame_buffer_info,
                                      (const graphics_image_t*)efect_info.p_pic_now,
                                      (const graphics_image_t*)efect_info.p_pic_next,
                                      (float32_t)i / (float32_t)SCROLL_STEP_NUM, direction);
                }
                swap_file_buff();
                break;
            case EVENT_MOVE:
                if ((efect_info.x_move * SCROLL_DIRECTION) >= 0) {
                    direction = SCROLL_DIRECTION_NEXT;
                    read_next_file();
                } else {
                    direction = SCROLL_DIRECTION_PREV;
                    read_prev_file();
                }
                draw_image_scroll(&frame_buffer_info,
                                  (const graphics_image_t*)efect_info.p_pic_now,
                                  (const graphics_image_t*)efect_info.p_pic_next,
                                  (float32_t)abs(efect_info.x_move) / (float32_t)LCD_PIXEL_WIDTH, direction);
                break;
            case EVENT_MOVE_END:
                type = 0;
                if ((efect_info.x_move * SCROLL_DIRECTION) >= 0) {
                    direction = SCROLL_DIRECTION_NEXT;
                    read_next_file();
                } else {
                    direction = SCROLL_DIRECTION_PREV;
                    read_prev_file();
                }

                wk_abs_x_pos = abs(efect_info.x_move);
                if (abs(efect_info.swipe_end_move) > 10) {
                    if ((efect_info.swipe_end_move * SCROLL_DIRECTION) >= 0) {
                        if (direction != SCROLL_DIRECTION_NEXT) {
                            type = 1;
                        }
                    } else {
                        if (direction != SCROLL_DIRECTION_PREV) {
                            type = 1;
                        }
                    }
                } else {
                    if (wk_abs_x_pos < (LCD_PIXEL_WIDTH / 2)) {
                        type = 1;
                    }
                }

                if (type == 0) {
                    while (wk_abs_x_pos < LCD_PIXEL_WIDTH) {
                        wk_abs_x_pos += (LCD_PIXEL_WIDTH / SCROLL_STEP_NUM);
                        if (wk_abs_x_pos > LCD_PIXEL_WIDTH) {
                            wk_abs_x_pos = LCD_PIXEL_WIDTH;
                        }
                        draw_image_scroll(&frame_buffer_info,
                                          (const graphics_image_t*)efect_info.p_pic_now,
                                          (const graphics_image_t*)efect_info.p_pic_next,
                                          (float32_t)wk_abs_x_pos / (float32_t)LCD_PIXEL_WIDTH, direction);
                    }
                    swap_file_buff();
                } else {
                    while (wk_abs_x_pos > 0) {
                        wk_abs_x_pos -= (LCD_PIXEL_WIDTH / SCROLL_STEP_NUM);
                        if (wk_abs_x_pos < 0) {
                            wk_abs_x_pos = 0;
                        }
                        draw_image_scroll(&frame_buffer_info,
                                          (const graphics_image_t*)efect_info.p_pic_now,
                                          (const graphics_image_t*)efect_info.p_pic_next,
                                          (float32_t)wk_abs_x_pos / (float32_t)LCD_PIXEL_WIDTH, direction);
                    }
                }
                efect_info.x_move = 0;
                break;
            case EVENT_NONE:
            case EVENT_DISP:
            default:
                if ((efect_info.scale_end_move_x != 0) || (efect_info.scale_end_move_y != 0) || (wk_event_req == EVENT_DISP)) {
                    if ((efect_info.scale_end_move_x != 0) || (efect_info.scale_end_move_y != 0)) {
                        efect_info.scale_end_move_x *= 0.8f;
                        efect_info.scale_end_move_y *= 0.8f;
                        zoom_scroll(efect_info.scale_end_move_x, efect_info.scale_end_move_y);
                        if (abs(efect_info.scale_end_move_x) < 5) {
                            efect_info.scale_end_move_x = 0;
                        }
                        if (abs(efect_info.scale_end_move_y) < 5) {
                            efect_info.scale_end_move_y = 0;
                        }
                    }
                    draw_image(&frame_buffer_info,
                               (const graphics_image_t*)efect_info.p_pic_now,
                               efect_info.drow_pos_x,
                               efect_info.drow_pos_y,
                               efect_info.magnification);
                    wait_time = 0;
                } else if (((file_id_now == 0xffffffff) && (wait_time >= 3000))
                        || ((disp_wait_time != 0) && (wait_time >= disp_wait_time) && (mouse_info.disp_pos == false))) {
                    efect_info.magnification = 1.0f;
                    read_next_file();
                    dissolve_seq = 1;
                } else {
                    /* do nothing */
                }
                if (dissolve_seq != 0) {
                    draw_image_dissolve(&frame_buffer_info,
                                        (const graphics_image_t*)efect_info.p_pic_now,
                                        (const graphics_image_t*)efect_info.p_pic_next,
                                        (float32_t)dissolve_seq / (float32_t)DISSOLVE_STEP_NUM);
                    if (dissolve_seq >= DISSOLVE_STEP_NUM) {
                        dissolve_seq = 0;
                        swap_file_buff();
                    } else {
                        dissolve_seq++;
                    }
                    wait_time = 0;
                }
                break;
        }
        if ((mouse_info.disp_time != false) && (system_timer.read_ms() > 1500)) {
            mouse_info.disp_time = false;
            if ((efect_info.event_req == EVENT_NONE) && (dissolve_seq == 0)) {
                draw_image(&frame_buffer_info,
                           (const graphics_image_t*)efect_info.p_pic_now,
                           efect_info.drow_pos_x,
                           efect_info.drow_pos_y,
                           efect_info.magnification);
            }
        }
        if ((wk_event_req != EVENT_NONE) || (touch_key_in != false) || (efect_info.magnification != 1.0f) || (dissolve_seq != 0)
         || (mouse_info.disp_pos != false) || (mouse_info.disp_time != false) || (efect_info.scroll != false)) {
            wait_time = 0;
        } else {
            Thread::wait(50);
            if (wait_time < 100000) {
                wait_time += 50;
            }
        }
    }
}
