#include "mbed.h"
#include "utils.hpp"

#include "EthernetInterface.h"
#include "frdm_client.hpp"

#include "metronome.hpp"

#define IOT_ENABLED

namespace active_low
{
	const bool on = false;
	const bool off = true;
}

DigitalOut g_led_red(LED1);
DigitalOut g_led_green(LED2);
DigitalOut g_led_blue(LED3);

InterruptIn g_button_mode(SW3);
InterruptIn g_button_tap(SW2);

metronome *m = new metronome();

float delay = 0, pulse = 0.1f;

void on_mode()
{
    // Change modes
    if (m->is_timing()) {
    	m->stop_timing();	
    } else {
    	m->start_timing();
    }

}

void on_tap()
{
    if(m->is_timing()){
    	g_led_red = active_low::on;
	    wait(pulse);
	    g_led_red = active_low::off;
	    m->tap();
	}
}

int main()
{
	// Seed the RNG for networking purposes
    unsigned seed = utils::entropy_seed();
    srand(seed);

	// LEDs are active LOW - true/1 means off, false/0 means on
	// Use the constants for easier reading
    g_led_red = active_low::off;
    g_led_green = active_low::off;
    g_led_blue = active_low::off;

	// Button falling edge is on push (rising is on release)
    g_button_mode.fall(&on_mode);
    g_button_tap.fall(&on_tap);
    
    

#ifdef IOT_ENABLED
	// Turn on the blue LED until connected to the network
    g_led_blue = active_low::on;

	// Need to be connected with Ethernet cable for success
    EthernetInterface ethernet;
    if (ethernet.connect() != 0)
        return 1;

	// Pair with the device connector
    frdm_client client("coap://api.connector.mbed.com:5684", &ethernet);
    if (client.get_state() == frdm_client::state::error)
        return 1;

	// The REST endpoints for this device
	// Add your own M2MObjects to this list with push_back before client.connect()
    M2MObjectList objects;
    

    M2MDevice* device = frdm_client::make_device();
    objects.push_back(device);

    
    M2MObject* frequency_object = M2MInterfaceFactory::create_object("3318");
    M2MObjectInstance* frequency_object_instance = frequency_object->create_object_instance();
    
    M2MResource* set_point_value = frequency_object_instance->create_dynamic_resource("5900", "Set",
    	M2MResourceInstance::STRING, true);
    set_point_value->set_operation(M2MBase::GET_PUT_ALLOWED);
    set_point_value->set_value((uint8_t*)"0", 1);
    
    M2MResource* min = frequency_object_instance->create_dynamic_resource("5601", "Min",
    	M2MResourceInstance::STRING, true);
	min->set_operation(M2MBase::GET_PUT_ALLOWED);
	min->set_value((uint8_t*)"0", 1);
	
	M2MResource* max = frequency_object_instance->create_dynamic_resource("5602", "Max",
		M2MResourceInstance::STRING, true);
	max->set_operation(M2MBase::GET_PUT_ALLOWED);
	max->set_value((uint8_t*)"0", 1);
	
	M2MResource* units = frequency_object_instance->create_dynamic_resource("5701", "Units",
		M2MResourceInstance::STRING, true);
	units->set_operation(M2MBase::GET_PUT_ALLOWED);
	units->set_value((uint8_t*)"BPM", 3);

	objects.push_back(frequency_object);
	
	
	// Publish the RESTful endpoints
    client.connect(objects);

	// Connect complete; turn off blue LED forever
    g_led_blue = active_low::off;
#endif

    while (true)
    {
#ifdef IOT_ENABLED
        if (client.get_state() == frdm_client::state::error)
            break;
#endif
		
        // Insert any code that must be run continuously here
        
    	char buffer[20];
        int size;
        if(!m->is_timing()){
            size = sprintf(buffer,"%d",m->get_bpm());
            set_point_value->set_value((uint8_t*)buffer, size);
        	delay = m->get_delay() / 1000.0f;
        	if(delay != 0){
        		g_led_green = active_low::on;
	        	wait(pulse);
	        	g_led_green = active_low::off;
        		wait(delay - pulse);	
        	}        		
        }
    }

#ifdef IOT_ENABLED
    client.disconnect();
#endif

    return 1;
}
