/**********************************************************************************************
 Copyright (c) 2014 DisplayModule. All rights reserved.

 Redistribution and use of this source code, part of this source code or any compiled binary
 based on this source code is permitted as long as the above copyright notice and following
 disclaimer is retained.

 DISCLAIMER:
 THIS SOFTWARE IS SUPPLIED "AS IS" WITHOUT ANY WARRANTIES AND SUPPORT. DISPLAYMODULE ASSUMES
 NO RESPONSIBILITY OR LIABILITY FOR THE USE OF THE SOFTWARE.
 ********************************************************************************************/

/******************************************************************************
 * Includes
 *****************************************************************************/

#include "mbed.h"

#include "DmTftIli9341.h"
#include "DmTftSsd2119.h"
#include "DmTftRa8875.h"
#include "DmTouch.h"

#include "W25Q16BV.h"
#include "DmDrawBmpBase.h"
#include "SDFileSystem.h"

/******************************************************************************
 * Typedefs and defines
 *****************************************************************************/

/* Note that there are restrictions on which platforms that can use printf
   in combinations with the DmTftLibrary. Some platforms (e.g. LPC1549 LPCXpresso)
   use the same pins for USBRX/USBTX and display control. Printing will
   cause the display to not work. Read more about this on the display's notebook
   page. */
#define log(...) printf(__VA_ARGS__)
//#define log(...)

#define DM_PIN_SPI_MOSI   D11
#define DM_PIN_SPI_MISO   D12
#define DM_PIN_SPI_SCLK   D13

#define DM_PIN_CS_TOUCH   D4
#define DM_PIN_CS_TFT     D10
#define DM_PIN_CS_SDCARD  D8
#define DM_PIN_CS_FLASH   D6

/******************************************************************************
 * Local variables
 *****************************************************************************/

//DmTftIli9341 tft;  /* DM_TFT28_105 */
//DmTftSsd2119 tft;   /* DM_TFT35_107 */
DmTftRa8875  tft;  /* DM_TFT43_108 and DM_TFT50_111   For DmTftRa8875 driver, The panel resolution should be config in DmTftRa8875::init() function on the DmTftRa8875.cpp file. */

//DmTouch touch(DmTouch::DM_TFT28_105);
//DmTouch touch(DmTouch::DM_TFT35_107);

SDFileSystem sd(DM_PIN_SPI_MOSI, DM_PIN_SPI_MISO, DM_PIN_SPI_SCLK, DM_PIN_CS_SDCARD, "sd"); // mosi,miso,clk,cs
W25Q16BV flash(DM_PIN_SPI_MOSI, DM_PIN_SPI_MISO, DM_PIN_SPI_SCLK, DM_PIN_CS_FLASH); // mosi,miso,clk,cs

DigitalInOut csTouch(DM_PIN_CS_TOUCH, PIN_OUTPUT, PullUp, 1);
DigitalInOut csDisplay(DM_PIN_CS_TFT, PIN_OUTPUT, PullUp, 1);
DigitalInOut csSDCard(DM_PIN_CS_SDCARD, PIN_OUTPUT, PullUp, 1);
DigitalInOut csFlash(DM_PIN_CS_FLASH, PIN_OUTPUT, PullUp, 1);


/******************************************************************************
 * Global variables
 *****************************************************************************/

/******************************************************************************
 * Local functions
 *****************************************************************************/

static bool sdcardReader(uint32_t userData, uint8_t* data, int offset, int numBytes) {
  FILE* fp = (FILE*)userData;
  fseek(fp, offset, SEEK_SET);
  fread(data, 1, numBytes, fp);
  return true;
}

static bool spiFlashReader(uint32_t userData, uint8_t* data, int offset, int numBytes) {
  W25Q16BV* spiFlash = (W25Q16BV*)userData;
  spiFlash->readStream(offset, (char*)data, numBytes);
  return true;
}

static void copyImageToSPIFlash(const char* fname) {
  FILE* fp = fopen(fname, "r");
  int num, offset;
  uint8_t buff[128];
  
  do {
    if (fp == NULL) {
      log("Failed to open %s, so assuming SPI flash is correct\n", fname);
      break;
    } 
  
    // simple comparison of first 64 bytes in file and SPI flash
    // to avoid having to write to the flash on every boot
    num = fread(buff, 1, 64, fp);
    if (num == 64) {
      flash.readStream(0, (char*)&buff[64], 64);
      if (memcmp(buff, &buff[64], 64) == 0) {
        log("SPI flash content propably correct, not replacing it\n");
        break;
      }
    }
    fseek(fp, 0, SEEK_SET);
    
    log("Found %s, erasing SPI flash\n", fname);
    flash.chipErase();

    log("Copying %s to SPI flash\n", fname);
    offset = 0;
    while ((num = fread(buff, 1, 128, fp)) > 0) {
      flash.writeStream(offset, (char*)buff, num);
      offset+=num;
    }
    log("Done. Wrote %d bytes to SPI flash\n", offset);
    
  } while(false);
  
  if (fp != NULL) {
    fclose(fp);
  }
}

/******************************************************************************
 * Main
 *****************************************************************************/

int main() {
  DmDrawBmpBase bmp;
  const char* fname = "/sd/logop565.bmp";

  copyImageToSPIFlash(fname);
  
  log("init tft \r\n");
  tft.init();
  
  while (true) {
    log("Drawing from SD card (%s) ...\n", fname);
    FILE *fp = fopen(fname, "r");
    if (fp != NULL) {
      bmp.drawBitmap(tft, 0, 0, sdcardReader, (uint32_t)fp);
      fclose(fp);
    } else {
      log("failed to open file\n");
      log("Skipping SDCard reading as it is unsupported\n");
      tft.drawStringCentered(0, 0, tft.width(), tft.height(), "SD Card Unsupported!");
    }

    wait_ms(3000);
    tft.clearScreen(RED);
    wait_ms(1000);

    log("Attempting to decode image from SPI flash...\n");
    if (!bmp.drawBitmap(tft, 0, 0, spiFlashReader, (uint32_t)&flash)) {
      log("Skipping SPI Flash reading as it is unsupported\n");
      tft.drawStringCentered(0, 0, tft.width(), tft.height(), "SPI Flash Unsupported!");
    }

    wait_ms(3000);
    tft.clearScreen(GREEN);
    wait_ms(1000);
  }
}
